/* Arg office software 
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package arg.money;
import java.util.*;
import java.io.*;

/** 
 *  [<code>ProjectData</code>]<br>
 *  Manage AnnualData and Subjects in one project.
 *  @author SAKURAI, Masashi
 *  @version 1.0
 */ 

public class ProjectData implements Serializable,NumberIdentical,ReservableObject {

    static final long serialVersionUID = -7499542006944819134L;

    int pid ;
    int curYear ;
    String name ;
    String comment ;
    int curReceiptNumber ;

    List annuals ;

    List subjects ;
    int curSubjectId = 0;

    List subjectGroups ;
    int curSubjectGroupId = 0;

    List subjectSubgroups ;
    int curSubjectSubgroupId = 0;
    
    transient boolean lock = false;

    /** create for the first time */
    public ProjectData(int pid,String name,String comment) {
	this();
	this.pid = pid;
	this.name = name;
	this.comment = comment;
	this.curYear = SystemUtil.getYear();
	makeNewAnnualData(getCurYear());
    }

    /** copy */
    protected ProjectData(ProjectData pd) {
	this();
	this.pid = pd.getId();
	this.name = pd.getName();
	this.comment = pd.getComment();
	this.curYear = pd.getCurYear();
    }

    public int getId() {return pid;}

    public void setCurYear(int a) { curYear = a; }
    public int getCurYear() { return curYear; }

    public void setName(String a) { name = a; }
    public String getName() { return name; }
    public String toString() { return getName();}

    public void setComment(String a) { comment = a; }
    public String getComment() { return comment; }

    public void setCurReceiptNumber(int a) { curReceiptNumber = a; }
    public int getCurReceiptNumber() { return curReceiptNumber; }

    public List getAnnuals() { return annuals; }
    public AnnualData makeNewAnnualData(int y) {
	if (getAnnual(y) != null) {
	    System.err.println(
		"Project "+getName()+" has already had annual data...");
	    return null;
	}
	AnnualData a = new AnnualData(this,y);
	synchronized(annuals) {
	    annuals.add(a);
	    Collections.sort(annuals,IdComparator.getInstance());
	}
	return a;
    }

    public List getSubjects() { return subjects; }
    public Subject addSubject(String name,String comment,int gid,int sgid) {
	Subject s = new Subject(curSubjectId,gid,sgid,name,comment);
	synchronized(this) {
	    curSubjectId++;
	    subjects.add(s);
	    Collections.sort(subjects,IdComparator.getInstance());
	    syncAnnualData();
	}
	return s;
    }
    public void removeSubject(int sid,SlipManager sm) throws RemovalException {
	Subject s = getSubject(sid);
	if (sm == null) throw new RemovalException("NullSlipManager");
	if (s == null) throw new RemovalException("CannotFindSubject");
	if (isExistSlip(sid,sm)) throw new RemovalException("ExistSlip"); 
	synchronized(this) {
	    subjects.remove(s);
	    syncAnnualData();
	}
    }
    private boolean isExistSlip(int sid,SlipManager sm) {
	List arg = sm.query(-1,getId());
	sm.queryBySubject(sid,sid,arg);
	return !sm.getList(arg).isEmpty();
    }

    public List getSubjectGroups() { return subjectGroups; }
    public SubjectGroup addSubjectGroup(String name,String comment,
					int balanceGroup) {
	SubjectGroup sg = new SubjectGroup(
	    curSubjectGroupId,name,comment,balanceGroup);
	synchronized(this) {
	    curSubjectGroupId++;
	    subjectGroups.add(sg);
	    Collections.sort(subjectGroups,IdComparator.getInstance());
	    syncAnnualData();
	}
	return sg;
    }
    public void removeSubjectGroup(int groupId,SlipManager sm) throws RemovalException {
	SubjectGroup s = getSubjectGroup(groupId);
	if (sm == null) throw new RemovalException("NullSlipManager");
	if (s == null) throw new RemovalException("CannotFindSubjctGroup");
	//empty check
	Iterator it = subjects.iterator();
	while(it.hasNext()) {
	    Subject sb = (Subject)it.next();
	    if (sb.getGroupId() == groupId) 
		if (isExistSlip(sb.getId(),sm))
		    throw new RemovalException("ExistSlip");
	}
	//remove subjects belonging to this group
	it = subjects.iterator();
	while(it.hasNext()) {
	    Subject sb = (Subject)it.next();
	    if (sb.getGroupId() == groupId) {
		it.remove();
	    }
	}
	//remove subgroups belonging to this group
	it = subjectSubgroups.iterator();
	while(it.hasNext()) {
	    SubjectSubgroup sb = (SubjectSubgroup)it.next();
	    if (sb.getGroupId() == groupId) {
		it.remove();
	    }
	}
	synchronized(this) {
	    subjectGroups.remove(s);
	    syncAnnualData();
	}
    }

    public List getSubjectSubgroups() { return subjectSubgroups; }
    public SubjectSubgroup addSubjectSubgroup(
	String name,String comment,int gid) {
	SubjectSubgroup sg = new SubjectSubgroup(curSubjectSubgroupId,gid,
						 name,comment);
	synchronized(this) {
	    curSubjectSubgroupId++;
	    subjectSubgroups.add(sg);
	    Collections.sort(subjectSubgroups,IdComparator.getInstance());
	    syncAnnualData();
	}
	return sg;
    }
    public void removeSubjectSubgroup(int subgroupId,SlipManager sm) throws RemovalException {
	SubjectSubgroup s = getSubjectSubgroup(subgroupId);
	if (s == null) throw new RemovalException("CannotFindSubgroup");
	if (sm == null) throw new RemovalException("NullSlipManager");

	boolean finish = false;
	while(!finish) {
	    finish = true;
	    Iterator it = subjects.iterator();
	    while(it.hasNext()) {
		Subject sb = (Subject)it.next();
		if (sb.getSubgroupId() == subgroupId) {
		    removeSubject(sb.getId(),sm);
		    finish = false;
		    break;
		}
	    }
	}
	synchronized(this) {
	    subjectSubgroups.remove(s);
	    syncAnnualData();
	}
    }

    /**
     * @param sid subject id
     * @return Subject object
     */ 
    public Subject getSubject(int sid) {
	return (Subject)(IdComparator.search(subjects,sid));
    }

    /**
     * @param y the year of the annual data
     * @return AnnualData object
     */ 
    public AnnualData getAnnual(int y) {
	return (AnnualData)(IdComparator.search(annuals,y));
    }

    /**
     * @param sid subject id
     * @return SubjectGroup object
     */ 
    public SubjectGroup getGroupBySubject(int sid) {
	Subject s = getSubject(sid);
	return getSubjectGroup(s.getGroupId());
    }

    /**
     * @param sid subject id
     * @return SubjectSubgroup object
     */ 
    public SubjectSubgroup getSubgroupBySubject(int sid) {
	Subject s = getSubject(sid);
	return getSubjectSubgroup(s.getSubgroupId());
    }

    /**
     * @param groupId subject group id
     * @return SubjectGroup object
     */ 
    public SubjectGroup getSubjectGroup(int groupId) {
	return (SubjectGroup)(IdComparator.search(subjectGroups,groupId));
    }

    /**
     * @param subgroupId subject subgroup id
     * @return SubjectSubgroup object
     */ 
    public SubjectSubgroup getSubjectSubgroup(int subgroupId) {
	return (SubjectSubgroup)
	    (IdComparator.search(subjectSubgroups,subgroupId));
    }

    /**
     * [<code>balanceSlips</code>]
     *  balance this year's slips and make next annual object.
     */ 
    public void balanceSlips(SlipManager sm) {
	int y = getCurYear();
	AnnualData curAnnual = getAnnual(y);
	makeNewAnnualData(y+1);
	AnnualData nextAnnual = getAnnual(y+1);
	Iterator it = getSubjects().iterator();
	while(it.hasNext()){
	    Subject sbj = (Subject)it.next();
	    //calculate slips
	    SubjectGroup sg = getGroupBySubject(sbj.getId());
	    //record
	    SubjectCurrentMoney scm = curAnnual.getCurrentMoney(sbj.getId());
	    CurrentAmountData am = scm.getAmounts(sm);
	    //set to the initial data of next year
	    if (!sg.isClear()) {
		SubjectMoney mm = nextAnnual.getInitialMoney(sbj.getId());
		mm.setMoney( am.getCurrentAmount() );
	    }
	}
	curAnnual.setClosedDate(new Date());
	setCurYear( y+1 );
    }

    /**
     * calculate summation
     *  
     * @param sid subject id
     * @param y year
     * @return CurrentAmountData
     */ 
    public CurrentAmountData calculate(SlipManager sm,int y,int sid) {
	AnnualData curAnnual = getAnnual(y);
	//calculate slips
	List arg = sm.query(y, getId());
	sm.queryBySubject(sid,sid,arg);
	List ss = sm.getList(arg);
	AmountData am = sm.calculate(ss,sid);
	//calculate total amount
	SubjectGroup sg = getGroupBySubject(sid);
	long tm = sg.getTotalAmount(
	    am.getCreditAmount(),am.getDebitAmount())+
	    curAnnual.getInitialMoney(sid).getMoney();
	return new CurrentAmountData(am,tm);
    }
    

    //=================================
    // private area
    //=================================

    public void syncAnnualData() {
        AnnualData ad = getAnnual(getCurYear());
        ad.sync();
    }

    /** constructor for loading */
    public ProjectData() {
	curReceiptNumber = 0;
	curYear = 0;
	annuals = new ArrayList();
	subjects = new ArrayList();
	subjectGroups = new ArrayList();
	subjectSubgroups = new ArrayList();
    }

    public void load(List args) {
	try {
	    if ( args.size() >= 12 ) {
		//decode slip data!!
		int ct = 0;
		pid = ((Integer)args.get(ct++)).intValue();
		setCurYear(((Integer)args.get(ct++)).intValue());
		setName((String)args.get(ct++));
		setComment((String)args.get(ct++));
		setCurReceiptNumber(((Integer)args.get(ct++)).intValue());
		annuals = (List)args.get(ct++);
		for (int i=0;i<annuals.size();i++)
		    ((AnnualData)annuals.get(i)).setParent(this);
		subjects = (List)args.get(ct++);
		curSubjectId = ((Integer)args.get(ct++)).intValue();
		subjectGroups = (List)args.get(ct++);
		curSubjectGroupId = ((Integer)args.get(ct++)).intValue();
		subjectSubgroups = (List)args.get(ct++);
		curSubjectSubgroupId = ((Integer)args.get(ct++)).intValue();
		Collections.sort(subjects,IdComparator.getInstance());
		Collections.sort(subjectSubgroups,IdComparator.getInstance());
		Collections.sort(subjectGroups,IdComparator.getInstance());
	    } 
	} catch(Exception e) {
	    //System.err.println(
	    //	e.getClass().getName()+" : "+e.getMessage());
	    e.printStackTrace();
	}
    }

    public List save() {
	List ret = new LinkedList();
	ret.add(new Integer(getId()));
	ret.add(new Integer(getCurYear()));
	ret.add(getName());
	ret.add(getComment());
	ret.add(new Integer(getCurReceiptNumber()));
	ret.add(annuals);
	ret.add(subjects);
	ret.add(new Integer(curSubjectId));
	ret.add(subjectGroups);
	ret.add(new Integer(curSubjectGroupId));
	ret.add(subjectSubgroups);
	ret.add(new Integer(curSubjectSubgroupId));
	return ret;
    }

    public ProjectData getCopy() {
	ProjectData copy = new ProjectData(this);
	copy.setCurYear(getCurYear());
	copy.setCurReceiptNumber(getCurReceiptNumber());
	//copy subject data
	Iterator it = subjects.iterator();
	while(it.hasNext()) {
	    Subject s = (Subject)it.next();
	    copy.subjects.add( s.getCopy() );
	}
	copy.curSubjectId = curSubjectId;
	//copy subject group
	it = subjectGroups.iterator();
	while(it.hasNext()) {
	    SubjectGroup sg = (SubjectGroup)it.next();
	    copy.subjectGroups.add( sg.getCopy() );
	}
	copy.curSubjectGroupId = curSubjectGroupId;
	//copy subject subgroup
	it = subjectSubgroups.iterator();
	while(it.hasNext()) {
	    SubjectSubgroup sg = (SubjectSubgroup)it.next();
	    copy.subjectSubgroups.add( sg.getCopy() );
	}
	copy.curSubjectSubgroupId = curSubjectSubgroupId;
	//copy annual data
	it = annuals.iterator();
	while(it.hasNext()) {
	    AnnualData an = (AnnualData)it.next();
	    copy.annuals.add( an.getCopy( copy ) );
	}
	return copy;
    }

    //===============================
    // LOCKING FRAMEWORK
    //===============================

    public synchronized boolean lock() {
	if (lock) return false;
	lock = true;
	return true;
    }

    public boolean isLocked() {
	return lock;
    }

    public synchronized void unlock() {
	lock = false;
    }
}
