/* Arg office software 
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package arg.money;
import java.util.*;
import java.io.*;

/** 
 *  [<code>ProjectManager</code>]<br>
 *  Manage ProjectDatas
 *  @author SAKURAI, Masashi
 *  @version 1.0
 */ 

public class ProjectManager extends AbstractManager {
    List projects ;
    int curPid = 0;

    /** start up and load files */
    public ProjectManager(SystemInfo p) {
	super(p,p.getProjectsFilename());
	projects = new ArrayList();
    }

    /**
     * [<code>getProjects</code>]
     * @return projects list
     */ 
    public List getProjects() { return projects; }

    /** make a ProjectData. after you call comitProjectData() to regist.
     * @param n project name
     * @param c project comment
     * @return temporary project object
     */ 
    public ProjectData makeProject(String n, String c) {
	ProjectData d = new ProjectData(curPid++,n,c);
	return d;
    }

    /**
     * remove project data
     * @param pid project id
     */ 
    public void removeProject(int pid) throws DataLockingException {
	ProjectData d = lockProjectData(pid);
	SlipManager sm = getParent().getSlipManager();
	if (d != null) {
	    d = getProjectData(pid);
	    projects.remove(d);
	    sm.removeProject(pid);
	}
	sm.modify(-1,-1);
    }

    /**
     * return ProjectData object as read-only.
     * This object was true object, not copy.
     * @param pid project id
     * @return project object
     */ 
    public ProjectData getProjectData(int pid) {
	return (ProjectData)(IdComparator.search(projects,pid));
    }

    /**
     * return ProjectData object as a copy.
     * If modified, call cancelProject() or comitProject().
     * @param pid project id
     * @return project object
     */ 
    public ProjectData lockProjectData(int pid) throws DataLockingException  {
	ProjectData pd = (ProjectData)(IdComparator.search(projects,pid));
	if (pd.lock()) {
	    //Success!
	    modify( pd.getCurYear() , pid );
	    return pd.getCopy();
	}
	//Already ready!
	throw new DataLockingException("ProjectDataLocked");
    }

    /** Unlock ProjectData object
     * @param pd ProjectData object as a copy
     */ 
    public void cancelProjectData(ProjectData pd) {
	ProjectData tpd = getProjectData(pd.getId());
	tpd.unlock();
	modify( tpd.getCurYear(), tpd.getId() );
    }

    /** Comit modification and unlock ProjectData object
     * @param pd modified ProjectData object 
     */ 
    public void comitProjectData(ProjectData pd) {
	ProjectData tpd = getProjectData(pd.getId());
	synchronized (this) {
	    if (tpd != null) {
		//edit
		if (tpd != pd) {
		//new object
		    tpd.unlock();
		    projects.remove(tpd);
		    projects.add(pd);
		} else
		    //continue object
		    pd.unlock();
	    } else {
		//create object
		projects.add(pd);
	    }
	    Collections.sort(projects,IdComparator.getInstance());
	}
	pd = getProjectData(pd.getId());
	modify( pd.getCurYear(), pd.getId() );
    }

    /** set modify frag and broadcast update message.
     * @param pid project id
     */
    public void modify(int year, int pid) {
	SlipManager sm = getParent().getSlipManager();
	sm.modify(year,pid);
    }

    public void makeNextYear(SlipManager slipManager) throws DataLockingException {
	Iterator it = projects.iterator();
	while(it.hasNext()) {
	    ProjectData sp = (ProjectData)it.next();
	    lockProjectData(sp.getId());
	    //
	    AnnualData ad = sp.getAnnual(sp.getCurYear());
	    ad.setClosedDate(new Date());
	    AnnualData nextAd = sp.makeNewAnnualData(sp.getCurYear()+1);
	    for (int i=0;i<ad.currentMoneys.size();i++) {
		SubjectCurrentMoney sm = 
		    (SubjectCurrentMoney)ad.currentMoneys.get(i);
		SubjectGroup sg = sp.getGroupBySubject(sm.getId());
		if (sg.getBalanceGroup() == SubjectGroup.ASSETS ||
		    sg.getBalanceGroup() == SubjectGroup.DEBT) {
		    SubjectMoney nsm = nextAd.getInitialMoney(sm.getId());
		    nsm.setMoney( sm.getCurrentAmount(slipManager) );
		}
	    }
	    sp.setCurYear(sp.getCurYear()+1);
	    //
	    cancelProjectData(sp);
	}
    }

    //=================================
    // private area
    //=================================

    public void loadClasses(ObjectInputStream in) throws Exception {
	//read ProjectManager property
	List property = (List)in.readObject();
	Integer ig = (Integer)property.get(0);
	curPid = ig.intValue();
	//project data reading
	List list = (List)in.readObject();
	Iterator it = list.iterator();
	while(it.hasNext()) {
	    List objs = (List)it.next();
	    ProjectData sp = new ProjectData();
	    sp.load(objs);
	    projects.add(sp);
	}
	Collections.sort(projects,IdComparator.getInstance());
    }

    protected void initByClean() {
	projects.clear();
	System.out.println("New project database was created.");
	ProjectData d = makeProject(
	    SystemUtil.getResourceString("DefaultProject"),
	    SystemUtil.getResourceString("DefaultProjectCommenct"));
	ProjectMaker.make(d,getParent().getProjectResourceFilename());
        comitProjectData( d );
	System.out.println("Made a default project.");
    }

    public void saveClasses(ObjectOutputStream out) throws Exception {
	//save SlipManager property
	List property = new LinkedList();
	property.add(new Integer(curPid));
	out.writeObject(property);
	//page writing
	LinkedList list = new LinkedList();
	Iterator it = projects.iterator();
	while(it.hasNext()) {
	    ProjectData sp = (ProjectData)it.next();
	    list.add(sp.save());
	}
	out.writeObject(list);
    }
}
