/* Arg office software 
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package arg.money;
import java.util.*;
import java.io.*;
import arg.money.gui.*;

/** 
 *  [<code>SecurityManager</code>]<br>
 *  Manage UserManager and GroupManager
 *  @author SAKURAI, Masashi
 *  @version 1.0
 */ 

public class SecurityManager extends AbstractManager {

    //Access constants
    public static final int READABLE = 1;//read slips and amount money
    public static final int WRITABLE = 2;//write/edit slips
    public static final int EDITABLE = 4;//edit project data
    public static final int SETTLABLE =8;//settle an account
    public static final int ADMIN =   16;//administer task

    UserManager userManager;
    GroupManager groupManager;
    SessionManager sessionManager;

    /** 
     *@param s parent SystemInfo object
     *@param f security data filename.
     */
    private SecurityManager(SystemInfo s,String f) {
	super(s,f);
	userManager = new UserManager(this);
	groupManager = new GroupManager(this);
    }

    /** 
     *@param s parent SystemInfo object
     */
    public SecurityManager(SystemInfo p) {
	this(p,p.getSecurityFilename());
    }

    public UserManager getUserManager() {return userManager;}
    public GroupManager getGroupManager() {return groupManager;}

    /**
     * @param name user name
     * @param passwd password
     * @return user id. If not certified, return -1.
     */ 
    public int login(String name,String passwd) {
	return getUserManager().login(name,passwd);
    }

    /**
     * @param session session objet
     * @param pid project id
     * @param task task id (READABLE,WRITABLE,EDITABLE and SETTLABLE)
     * @return if true, allow the operation.
     */ 
    public boolean getPermission(SessionData session, int pid,int task) {
	if (!sessionManager.checkSession(session)) return false;
	return getPermission(session.getUid(),pid,task);
    }

    /**
     * @param session session object
     * @return if true, allow the operation.
     */ 
    public boolean getAdminPermission(SessionData session) {
	if (session == null) return false;
	if (!sessionManager.checkSession(session)) return false;
	return getAdminPermission(session.getUid());
    }
    /**
     * @param uid user id
     * @param pid project id
     * @param task task id (READABLE,WRITABLE,EDITABLE and SETTLABLE)
     * @return if true, allow the operation.
     */ 
    protected boolean getPermission(int uid, int pid,int task) {
	return getGroupManager().getPermission(uid,pid,task);
    }

    /**
     * @param uid user id
     * @return if true, allow the operation.
     */ 
    protected boolean getAdminPermission(int uid) {
	return getGroupManager().getAdminPermission(uid);
    }

    //=================================
    // utility area
    //=================================

    /** 
     *@param  uid user id
     *@param  task task id (READABLE,WRITABLE,EDITABLE and SETTLABLE)
     *@return list of ProjectData object
     */
    public List getPermittedProjects(int uid,int task) {
    	List projects = getParent().getProjectManager().getProjects();
	Iterator it = projects.iterator();
	List avlProjects = new LinkedList();
	while(it.hasNext()) {
	    ProjectData p = (ProjectData)it.next();
	    if (getPermission(uid,p.getId(),task))
		avlProjects.add(p);
	}
	return avlProjects;
    }

    /** 
     *@param  uid user id
     *@return list GroupData object
     */
    public List getBelongingGroups(int uid) {
	List groups = getGroupManager().getGroups();
	Iterator it = groups.iterator();
	List avlGroups = new LinkedList();
	while(it.hasNext()) {
	    GroupData p = (GroupData)it.next();
	    if (p.isBelonging(uid))
		avlGroups.add(p);
	}
	return avlGroups;
    }

    /** Update security data */
    public void update() {
	getParent().getSlipManager().modify(-1,-1);
    }

    //=================================
    // private area
    //=================================

    public void onStartup() {
	sessionManager = getParent().getSessionManager();
    }

    //test method
    public void main(String [] args) {
	SecurityManager sm = new SecurityManager(null,"test_security.dat");
	sm.initByLoad();
    }

    public void initByClean() {
	userManager.initByClean();
	groupManager.initByClean();
	System.out.println("New security database was created.");
    }

    public void loadClasses(ObjectInputStream in) throws Exception {
	//read SecurityManager property
	List property = (List)in.readObject();
	//user data reading
	List list = (List)in.readObject();
	userManager.loadClasses(list);
	//group data reading
	list = (List)in.readObject();
	groupManager.loadClasses(list);
    }

    public void saveClasses(ObjectOutputStream out) throws Exception {
	//save SlipManager property
	List property = new LinkedList();
	out.writeObject(property);
	//user data
	out.writeObject(userManager.saveClasses());
	out.writeObject(groupManager.saveClasses());
    }

}
