/* Arg office software 
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package arg.money;
import java.util.*;
import java.io.*;
import arg.money.gui.*;
import javax.swing.*;
import java.awt.*;
import java.util.List;
import java.text.MessageFormat;
import jp.gr.java_conf.ccs2.io.*;
import jp.gr.java_conf.ccs2.core.*;
import jp.gr.java_conf.ccs2.frame.*;

/** 
 *  [<code>SessionManager</code>]<br>
 *  
 *  @author SAKURAI, Masashi
 *  @version 1.0
 */ 

public class SessionManager extends AbstractManager {

    List sessions ;
    int curSessionId = 0;
    List logs;
    int maxLog = 300;
    View view;
    
    static final String openLog = "+ : {0} opened from {1} at {2}";
    static final String closeLog = "- : {0} closed from {1} at {2}";
    static final String failLog = "* : {0} login failed from {1} at {2}";
    static final String invalidSessionLog = "! : {0} invalid session from {1} at {2}";

    public SessionManager(SystemInfo s) {
	super(s,s.getSessionLogFilename());
	sessions = new ArrayList();
    }

    //=================================
    // operation
    //=================================

    public void login(RuntimeInformation rt) {
	if (rt == null) return;
	LoginShell dlg = rt.getShellFactory().getLoginShell();
	while(true) {
	    dlg.show();
	    if (dlg.isOk()) {
		int id = getParent().getSecurityManager().
		    login(dlg.getName(),dlg.getPassword());
		if (id == -1) {
		    rt.getShellFactory().showMessage(
			SystemUtil.getResourceString("InvalidPassword"));
		    log(failLog,rt,dlg.getName());
		    continue;
		}
		addSession(rt,id);
	    }
	    break;
	}
    }

    public boolean checkSession(SessionData session) {
	try {
	    if (session == null) return false;
	    SessionData mine = getSessionData(session.getId());
	    if (mine == null) {
		UserData user = session.getUser();
		String uname = "(unknown)";
		if (user != null) {
		    uname = user.getName();
		    log(invalidSessionLog,
			session.getRuntimeInformation(), uname);
		    return false;
		}
	    }
	    if (mine.getUid() != session.getUid()) {
		return false;
	    }
	    return true;
	}catch (RuntimeException e) {
	    log("Error : "+e.getClass().getName()+" ["+e.getMessage()+"]");
	    e.printStackTrace();
	}
	return false;
    }
    
    public SessionData addSession(RuntimeInformation rt,int uid) {
	if (getSessionDataByUser(uid) != null) {
	    rt.getShellFactory().showMessage(
		SystemUtil.getResourceString("HadBeenLogin"));
	    return null;
	}
	SessionData sd = new SessionData(uid,curSessionId++,
					 this,rt);
	sessions.add(sd);
	Collections.sort(sessions,IdComparator.getInstance());
	// log
	UserData user = sd.getUser();
	if (user != null) 
	    log(openLog,sd.getRuntimeInformation(),user.getName());
	else {
	    log(invalidSessionLog,sd.getRuntimeInformation(),"<unknown>");
	    log("Connection refused.");
	    sessions.remove(sd);
	    return null;
	}
	sd.startup();

	return sd;
    }

    public void removeSession(int ssid) {
	SessionData sd = getSessionData(ssid);
	if (sd == null) return;
	sessions.remove(sd);
	// log
	UserData user = sd.getUser();
	if (user != null) 
	    log(closeLog,sd.getRuntimeInformation(),user.getName());
    }

    //=================================
    // access methods
    //=================================

    public List getSessions() {return sessions;}

    public View getView() {
	if (view == null) throw new NullPointerException();
	return view;
    }

    public SessionData getSessionData(int ssid) {
	return (SessionData)(IdComparator.search(sessions,ssid));
    }

    public SessionData getSessionDataByUser(int uid) {
	Iterator it = sessions.iterator();
	while(it.hasNext() ) {
	    SessionData sd = (SessionData)it.next();
	    if (sd.getUid() == uid) return sd;
	}
	return null;
    }

    //=================================
    // private area
    //=================================

    void log(String format,RuntimeInformation rt,String help) {
	String point = "(null)";
	if (rt != null) {
	    point = rt.getAccessPoint();
	}
	String [] arg = {help, point,
			 jp.gr.java_conf.ccs2.util.StringUtil.dateTimej(new Date())};
	log( MessageFormat.format(format,arg) );
    }

    public void log(String message) {
	logs.add(message);
	System.out.println(message);
	if (maxLog < logs.size())
	    logs.remove(0);
    }

    public void onStartup() {
	SlipManager sm = getParent().getSlipManager();
	view = new SessionView(sm.getDocument());
    }

    public void onExit() {
	while(sessions.size() > 0 ) {
	    SessionData sd = (SessionData)sessions.get(0);
	    sd.terminate();
	}
        super.onExit();
    }
    public void loadClasses(ObjectInputStream in) throws Exception {
	logs = (List)in.readObject();
    }

    public void initByClean() {
	logs = new LinkedList();
    }

    public void saveClasses(ObjectOutputStream out) throws Exception {
	out.writeObject(logs);
	//
	EasyWriter wt = new EasyWriter("session_log.txt");
	Iterator it = logs.iterator();
	while(it.hasNext()) {
	    String s = (String)it.next();
	    wt.println( s );
	}
	
	wt.flush();
    }

    //=====(CView implementaion)===

    class SessionView extends CView {
	public SessionView(Document d) {
	    super(d);
	}
	
	public void update(Object obj) {
	    int pid = ((Integer)obj).intValue();
	    updateSessions(pid);
	}
    }

    void updateSessions(int pid) {
	Iterator it = getSessions().iterator();
	while(it.hasNext()){
	    SessionData sd = (SessionData)it.next();
	    sd.updateData(pid);
	}
    }

}
