/* Arg office software 
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package arg.money;
import java.util.*;
import java.io.*;
import jp.gr.java_conf.ccs2.frame.*;

/** 
 *  [<code>SlipManager</code>]<br>
 *  Manage transfer slips
 *  @author SAKURAI, Masashi
 *  @version 1.0
 */ 

public class SlipManager extends AbstractManager{

    List slipBooks = new LinkedList();
    Document document = new Document();

    //===(SlipBook)==========================

    class SlipBook implements Comparator{
	int pid;
	int year;
	int curSlipId = 0;
	List slips;
	Date lastModified;

	/** called in loadSequence */
	SlipBook() {
	    slips = new ArrayList();
	    lastModified = new Date();
	}

	/** normal constructor  */
	SlipBook(int year,int pid) {
	    this();
	    setYear(year);
	    setPid(pid);
	}
	
	void loadClasses(List arg) throws Exception {
	    Iterator it = arg.iterator();
	    setYear( ( (Integer)it.next() ).intValue() );
	    setPid( ( (Integer)it.next() ).intValue() );
	    setCurSlipId( ( (Integer)it.next() ).intValue() );
	    lastModified = (Date)it.next();
	    while(it.hasNext()) {
		List objs = (List)it.next();
		int tmp = ((Integer)objs.get(0)).intValue();
		TransferSlip sp = new TransferSlip();
		sp.load(objs);
		slips.add(sp);
	    }
	    modify();
	}

	List saveClasses() throws Exception {
	    LinkedList list = new LinkedList();
	    list.add( new Integer(getYear()));
	    list.add( new Integer(getPid()));
	    list.add( new Integer(getCurSlipId()));
	    list.add( lastModified );
	    Iterator it = slips.iterator();
	    while(it.hasNext()) {
		TransferSlip sp = (TransferSlip)it.next();
		list.add(sp.save());
	    }
	    return list;
	}

	void setPid(int i) {pid = i;}
	void setYear(int i) {year = i;}
	void setCurSlipId(int i) {curSlipId = i;}
	int getPid() {return pid;}
	int getYear() {return year;}
	int getCurSlipId() {return curSlipId;}
	List getSlips() {return slips;}
	Date getLastModified() {return lastModified;}

	void modify() {
	    Collections.sort(slips,this);
	    lastModified.setTime(System.currentTimeMillis());
	}

	synchronized void addSlip(TransferSlip ts) {
	    TransferSlip cts = getSlip(ts.getId());
	    if (cts != null) {
		removeSlip(cts.getId());
	    }
	    slips.add(ts);
	    modify();
	}

	synchronized TransferSlip makeSlip(ProjectData pd,int u) {
	    TransferSlip ts = new TransferSlip(curSlipId,pd.getCurYear(),
					       pd.getCurReceiptNumber(),
					       pd.getId(),u,null);
	    curSlipId++;
	    return ts;
	}
	
	synchronized void removeSlip(int tid) {
	    TransferSlip ts =getSlip(tid);
	    if (ts == null) return;
	    modify();
	    slips.remove(ts);
	}
	
	TransferSlip getSlip(int tid) {
	    return (TransferSlip)(IdComparator.search(getSlips(),tid));
	}

    //comparator stuff
    public boolean equals(Object obj) {
	return super.equals(obj);
    }

    //comparator stuff
    public int compare(Object o1,Object o2) {
	if (o1 instanceof TransferSlip && o2 instanceof TransferSlip) {
	    TransferSlip t1 =(TransferSlip)o1;
	    TransferSlip t2 =(TransferSlip)o2;
	    if (t1.getDate().getTime() < t2.getDate().getTime()) 
		return -1;
	    if (t1.getDate().getTime() == t2.getDate().getTime()) 
		return 0;
	    return 1;
	}
	return 0;
    }

/*
	//comparator stuff
	public boolean equals(Object obj) {
	    return equals(obj);
	}
	//comparator stuff
	public int compare(Object o1,Object o2) {
	    if (o1 instanceof TransferSlip && o2 instanceof TransferSlip) {
		TransferSlip t1 =(TransferSlip)o1;
		TransferSlip t2 =(TransferSlip)o2;
		if (t1.getId() < t2.getId()) 
		    return -1;
		if (t1.getId() == t2.getId()) 
		    throw new InternalError("ID crossed.");
		return 1;
	    }
	    return 0;
	}
*/

    } // end of SlipBook
        
    private List getSlipBooks() {
	return slipBooks;
    }

    private SlipBook getSlipBook(int year,int pid) {
	Iterator it = slipBooks.iterator();
	while(it.hasNext()) {
	    SlipBook sb = (SlipBook)it.next();
	    if (sb.getYear() == year && sb.getPid() == pid )
		return sb;
	}
	return addSlipBook(year,pid);
    }

    private SlipBook addSlipBook(int year,int pid) {
	SlipBook sb = new SlipBook(year,pid);
	slipBooks.add(sb);
	return sb;
    }

    //called by ProjctData to remove 
    /*
      void removeSlipBook(int y,int p) {
      SlipBook b = getSlipBook(y,p);
      if (b != null)
      slipBooks.remove(b);
      }
    */

    //called by ProjctManager to remove ProjectData
    void removeProject(int pid) {
	Iterator it = slipBooks.iterator();
	while(it.hasNext()) {
	    SlipBook sb = (SlipBook)it.next();
	    if ( sb.getPid() == pid )
		it.remove();
	}
    }


    //===(SlipManager)==========================

    /** startup contructor */
    public SlipManager(SystemInfo p) {
	super(p,p.getSlipsFilename());
    }

    public Document getDocument() {
	return document;
    }

    /** Get a TransferSlip object on Read-only mode.
     * If you will modify, usethe method "lockSlip()".
     * @param tid TransferSlip's id
     * @param pid Project id
     * @param year book year
     * @return TransferSlip object
     */ 
    public TransferSlip getSlip(int year,int pid,int tid) {
	return getSlipBook(year,pid).getSlip(tid);
    }

    /** get the slip object and lock it for a while.
     * If finished, call the method "comitSlip()" or "cancelSlip()"
     * @param tid TransferSlip's id
     * @param pid Project id
     * @param year book year
     * @param time locking time (if -1, default time(60minutes) will be set.)
     * @return TransferSlip copy object. If not found, return null.
     * @exception DataLockingException data locked
     */ 
    public TransferSlip lockSlip(int year,int pid,int tid) throws DataLockingException {
	TransferSlip ts = getSlipBook(year,pid).getSlip(tid);
	if (ts == null) {
	    System.err.println("Y:"+year+"  PID:"+pid+"  TID:"+tid);
	    System.err.println("null slip data?");
	    return null;
	}
	if ( ts.lock() ) {
	    //Success!
	    modify(year,pid);
	    return ts.getCopy();
	}
	//Already ready!
	throw new DataLockingException("TransferSlipLocked");
    }

    /** Unlock slip object
     * @param ts modified TransferSlip 
     */ 
    public void cancelSlip(TransferSlip ts) {
	TransferSlip tts = getSlipBook(
	    ts.getYear(),ts.getPid()).getSlip(ts.getId());
	if (!tts.isLocked()) {
	    SystemUtil.logMessage("Cancel TS failed. "+tts.toString());
	    return;
	}
	tts.unlock();
	modify( ts.getYear(), ts.getPid() );
    }

    /** Comit modification and unlock slip object
     * @param ts modified TransferSlip 
     */ 
    public void comitSlip(TransferSlip ts) {
	SlipBook sb = getSlipBook( ts.getYear(), ts.getPid() );
	synchronized (sb) {
	    sb.addSlip(ts);
	    ts.unlock();
	    ts.modify();
	}
	modify( ts.getYear(), ts.getPid() );
    }
    
    /** make transfer slip object. after modify this object, comit it.
     * @param y year
     * @param rn receipt number
     * @param p project id
     * @param u user id
     * @param lines slip line object list
     * @return temporary transfer slip object
     */
    public TransferSlip makeSlip(ProjectData pd,int u) {
	SlipBook sb = getSlipBook(pd.getCurYear(),pd.getId());
	return sb.makeSlip(pd,u);
    }

    /** remove transfer slip object
     * @param year year
     * @param pid project id
     * @param tid transfer slip id
     */
    public void removeSlip(int year,int pid,int tid) {
	getSlipBook(year,pid).removeSlip(tid);
	modify( year, pid );
    }

    /** remove all transfer slip objects which have the given 
     * Subject id. This method is called by ProjectData while 
     * removing a Subject object.
     * [Now disable]
     * @param year year (-1 : all year)
     * @param pid project id
     * @param tid subject id
     */
    /*
      public void removeSlips(int year,int pid,int sid) {
      List arg = query(year,pid);
      queryBySubject(sid,sid,arg);
      Iterator it = getList(arg).iterator();
      while (it.hasNext()) {
      TransferSlip ts = (TransferSlip)it.next();
      removeSlip(ts.getYear(),ts.getPid(),ts.getId());
      }
      }
    */

    /** set modify frag and broadcast update message.
     * @param year year (if negative, update all years.)
     * @param pid project id (if negative, update all projects.)
     */
    public void modify(int year,int pid) {
	if (year < 0 || pid < 0) {
	    //multiple update
	    Iterator it = slipBooks.iterator();
	    while(it.hasNext()) {
		SlipBook sb = (SlipBook)it.next();
		if ( sb.getPid() == pid || sb.getYear() == year ||
		     (year<0 && pid<0) ) {
		    sb.modify();
		}
	    }
	} else {
	    //single update
	    getSlipBook(year,pid).modify();
	}
	document.updateViews(new Integer(pid));
    }

    /** get last modified date
     * @param year year
     * @param pid project id
     * @return last modified date
     */
    public Date getLastModified(int year,int pid) {
	return getSlipBook(year,pid).getLastModified();
    }

    /** make a temporary query set.
     * @param year year (-1 : all)
     * @param pid project id (-1 : all)
     * @return temporary data, consisting of SlipWrapper.
     * you can get the list of TransferSlip with [getList(list)]
     */ 
    public List query(int year,int pid) {
	List sample = new LinkedList();
	if (year >= 0) {//YEAR 
	    if (pid >= 0) {//PID
		sample.addAll(getSlipBook(year,pid).getSlips());
	    } else {//NO PID
		Iterator it = getSlipBooks().iterator();
		while(it.hasNext()) {
		    SlipBook sb = (SlipBook)it.next();
		    if (sb.getYear() == year)
			sample.addAll(sb.getSlips());
		}
	    }
	} else {//NO YEAR
	    if (pid >= 0) {//PID
		Iterator it = getSlipBooks().iterator();
		while(it.hasNext()) {
		    SlipBook sb = (SlipBook)it.next();
		    if (sb.getPid() == pid)
			sample.addAll(sb.getSlips());
		}
	    } else {//NO PID (ALL slips)
		Iterator it = getSlipBooks().iterator();
		while(it.hasNext()) {
		    SlipBook sb = (SlipBook)it.next();
		    sample.addAll(sb.getSlips());
		}
	    }
	}
	return query(sample);
    }

    /** make a temporary query set.
     * @param sample list of TransferSlip objects
     * @return temporary data, consisting of SlipWrapper.
     * you can get the list of TransferSlip with [getList(list)]
     */ 
    public List query(List sample) {
	if (sample == null) 
	    return null;
	Iterator it = sample.iterator();
	List ret = new LinkedList();
	while(it.hasNext()) {
	    TransferSlip sp = (TransferSlip)it.next();
	    ret.add(new SlipWrapper(sp,true));
	}
	return ret;
    }

    /**
     * transform temporary list to final result
     *  
     * @param arg   temporary data list
     * @return TransferSlip list
     */ 
    public List getList(List arg) {
	Iterator it = arg.iterator();
	List list = new LinkedList();
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    list.add( w.getSlip() );
	}
	return list;
    }


    /**
     * @param begin region of date query
     * @param end   region of date query
     * @param arg   temporary data list
     */ 
    public void queryByDate(Date begin,Date end,List arg) {
        Iterator it = arg.iterator();
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    if (begin != null) 
		if (w.getSlip().getDate().before(begin)) {
		    w.setSelected(false);
		    continue;
		}
	    if (end != null) 
		if (w.getSlip().getDate().after(end))
		    w.setSelected(false);
	}
    }

    /**
     * [<code>queryByGroup</code>]
     *  
     *  
     * @param gleft  left group id (-1:none)
     * @param gright right group id (-1:none)
     * @param arg   temporary data list
     */ 
    public void queryByGroup(int gleft, int gright, List arg) {
        Iterator it = arg.iterator();
	ProjectManager pm = getParent().getProjectManager();
    base:
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    List lines = w.getLines();
	    if (gleft != -1) {
		ProjectData pd = pm.getProjectData(w.getSlip().getPid());
		for (int i=0;i<lines.size();i++) {
		    ASlipLine line = (ASlipLine)lines.get(i);
		    SlipElement el = line.getLeft();
		    SubjectGroup sg = pd.getGroupBySubject(el.getId());
		    if ( sg.getId() == gleft ) continue base;
		}
	    }
	    if (gright != -1) {
		ProjectData pd = pm.getProjectData(w.getSlip().getPid());
		for (int i=0;i<lines.size();i++) {
		    ASlipLine line = (ASlipLine)lines.get(i);
		    SlipElement el = line.getRight();
		    SubjectGroup sg = pd.getGroupBySubject(el.getId());
		    if ( sg.getId() == gright ) continue base;
		}
	    }
	    w.setSelected(false);
	}
    }

    /**
     * @param sleft  subject left id
     * @param sright subject right id
     * @param arg   temporary data list
     */ 
    public void queryBySubject(int sleft, int sright, List arg) {
        if (sleft == -1 && sright == -1) return;
        //
        Iterator it = arg.iterator();
        if (sleft != -1 && sright != -1) {
            while(it.hasNext()) {
                SlipWrapper w = (SlipWrapper)it.next();
                if (!w.isSelected()) continue;
                w.setSelected(
                    w.getSlip().existsSlipLineByLeftSubject(sleft) ||
                    w.getSlip().existsSlipLineByRightSubject(sright)
                    );
            }
            return;
        }
        if (sleft != -1) {
            while(it.hasNext()) {
                SlipWrapper w = (SlipWrapper)it.next();
                if (!w.isSelected()) continue;
                w.setSelected(
                    w.getSlip().existsSlipLineByLeftSubject(sleft));
            }
            return;
        }

        while(it.hasNext()) {
            SlipWrapper w = (SlipWrapper)it.next();
            if (!w.isSelected()) continue;
            w.setSelected(
                w.getSlip().existsSlipLineByRightSubject(sright));
        }
    }

    /**
     * @param s summary string
     * @param arg   temporary data list
     */ 
    public void queryBySummary(String s, List arg) {
	if (s == null) return;
	if (s.equals("")) return;
        Iterator it = arg.iterator();
    base:
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    List lines = w.getLines();
	    for (int i=0;i<lines.size();i++) {
		ASlipLine line = (ASlipLine)lines.get(i);
		if ( line.getSummary().indexOf(s) != -1 ) continue base;
	    }
	    w.setSelected(false);
	}
    }

    /**
     * @param uid user id
     * @param arg   temporary data list
     */ 
    public void queryByUser(int uid, List arg) {
	if (uid == -1) return;
        Iterator it = arg.iterator();
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    if (w.getSlip().getUid() == uid) continue;
	    w.setSelected(false);
	}
    }

    /**
     * @param rep receipt nomber
     * @param arg   temporary data list
     */ 
    public void queryByReceipt(int rep, List arg) {
	if (rep == -1) return;
        Iterator it = arg.iterator();
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    if (w.getSlip().getReceiptNumber() == rep) continue;
	    w.setSelected(false);
	}
    }

    /**
     * @param mm money
     * @param arg   temporary data list
     */ 
    public void queryByMoney(long mm, List arg) {
	if (mm == -1) return;

        Iterator it = arg.iterator();
    base:
	while(it.hasNext()) {
	    SlipWrapper w = (SlipWrapper)it.next();
	    if (!w.isSelected()) continue;
	    List lines = w.getLines();
	    for (int i=0;i<lines.size();i++) {
		ASlipLine line = (ASlipLine)lines.get(i);
		if ( line.getLeft().getMoney() == mm ) continue base;
		if ( line.getRight().getMoney() == mm ) continue base;
	    }
	    w.setSelected(false);
	}
    }

    /**
     * calculate summation
     *  
     * @param sid subject id
     * @param arg TransferSlip list
     * @return AmountData object
     */ 
    public AmountData calculate(List arg,int sid) {
        Iterator it = arg.iterator();
	long left=0,right=0;
	while(it.hasNext()) {
	    TransferSlip slip = (TransferSlip)it.next();
	    List lines = slip.getLines();
	    for (int i=0;i<lines.size();i++) {
		ASlipLine line = (ASlipLine)lines.get(i);
		//left
		if (line.existLeft()) {
                    if (line.getLeft().getId() == sid || sid < 0)
    		        left += line.getLeft().getMoney();
		}
		//right
		if (line.existRight()) {
                    if (line.getRight().getId() == sid || sid < 0)
		        right += line.getRight().getMoney();
		}
	    }
	}
	return new AmountData(left,right);
    }

    //=================================
    // private area
    //=================================

    protected void loadClasses(ObjectInputStream in) throws Exception {
	//read SlipManager property
	List property = (List)in.readObject();
	//slip data reading
	List list = (List)in.readObject();
	Iterator it = list.iterator();
	while(it.hasNext()) {
	    List objs = (List)it.next();
	    SlipBook sp = new SlipBook();
	    sp.loadClasses(objs);
	    slipBooks.add(sp);
	}
    }

    protected void initByClean() {
	slipBooks.clear();
	System.out.println("New slip database was created.");
    }

    public void saveClasses(ObjectOutputStream out) throws Exception {
	//save SlipManager property
	List property = new LinkedList();
	out.writeObject(property);
	//page writing
	LinkedList list = new LinkedList();
	Iterator it = slipBooks.iterator();
	while(it.hasNext()) {
	    SlipBook sp = (SlipBook)it.next();
	    list.add(sp.saveClasses());
	}
	out.writeObject(list);
    }

    /** this class is a helper for query operation */
    class SlipWrapper {
	boolean selected;
	TransferSlip slip;
	SlipWrapper(TransferSlip slip,boolean s) {
	    this.slip = slip;
	    selected  = s;
	}
	boolean isSelected() {return selected;}
	void setSelected(boolean b) {selected = b;}
	TransferSlip getSlip() {return slip;}
	List getLines() {  return slip.getLines();}
    }

}
