/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.comp;
import jp.gr.java_conf.ccs2.util.ColorEditor;
import java.awt.*;
import java.awt.event.*;
import java.util.*;



/** Tab component (used in jp.gr.java_conf.ccs2.comp.TabPanel) */
public class TabCtrl extends Panel implements FontInfo,ColorInfo{

    Vector parents = new Vector();

    Hashtable table = new Hashtable();
    String activeTab = "";

    boolean isRepaint = false;

    int stringHeight = 15;
    int bottomOffset = 2;
    int topOffset = 1;
    int varticalOffset = 2;
    int horizontalOffset = 3;
    int delta = 2;
    int cutSize = 2;
    double spaceParam = 1.2;

    Color darkGray = Color.darkGray;
    Color lightGray = Color.white;
    Color gray = Color.gray;
    Color forground = Color.black;
    Color background = Color.lightGray;

    static final int DARKGRAY = BACKGROUND+1;
    static final int LIGHTGRAY = BACKGROUND+2;
    static final int GRAY = BACKGROUND+3;

    Font font1,font2;
    OriginalFont orgFont,sFont;

    //====== constructor

    public TabCtrl() {
	super();
	initFont();
	initLayout();
	setBackground(background);
    }

    public TabCtrl(MessageListener ms) {
	this();
	addMessageListener(ms);
    }

    //====== access method

    public void addMessageListener(MessageListener m) {
	parents.addElement(m);
    }

    public void setFont(int sz) {
	setFont(sz,Font.PLAIN);
    }
    public void setFont(int sz,int st) {
	stringHeight = (int)(sz*1.5);
	Font f = null;
	if (font1 == null) {
	    f = new Font("SansSerif",Font.PLAIN,sz);
	} else {
	    f = new Font(font1.getName(),font1.getStyle(),sz);
	}
	setFont(f,st);
    }

    public void setFont(Font f) {
	setFont(f,f.getStyle());
    }

    public void setFont(Font f,int selectedStyle) {
	font1 = f;
	font2 = new Font(f.getName(),selectedStyle,f.getSize());
	orgFont = new OriginalFont(forground,background,font1);
	sFont = new OriginalFont(forground,background,font2);
    }

    public void setColor(Color fc,Color bc) {
	forground = fc;
	background = bc;
	setBackground(bc);
	lightGray = ColorEditor.getBrighter(bc);
	darkGray = ColorEditor.getDarker(bc);
	orgFont.setColor(background ,forground );
	sFont.setColor(background,ColorEditor.getDarker(forground) );
	//updates (don't like)
	Enumeration e = table.elements();
	while(e.hasMoreElements()) {
	    ((TabItem)e.nextElement()).updateInfo();
	}
	updateImmediate();
    }

    //======== operation method

    public void setTabName(String key,String newName) {
	TabItem tb = (TabItem)table.get(key);
	if (tb == null) return;
	tb.setTabName(newName);
	isRepaint = true;
	updateImmediate();
    }

    public void addTab(String name,String key) {
	if (table.containsKey(key)) {
	    System.out.println("TabCtrl warning : the same key["+key+"]");
	    return;
	}
	TabItem item = new TabItem(name,key,this);
	add(item);
	table.put(key,item);
	if (table.size() == 1) {
	    item.setActive(true);
	    activeTab = key;
	    setActive(key);
	}
	isRepaint = true;
	TabItem pt = (TabItem)table.get(getActive());
	if (pt == null) {
	    getParent().doLayout();
	} else updateImmediate();
    }

    public String getActive() {
	return activeTab;
    }

    public void removeTab(String key) {
	TabItem tb = (TabItem)table.remove(key);
	if (tb != null) remove(tb);
	setActive(null);
	isRepaint = true;
    }

    public void removeAllTabs() {
	table.clear();
	removeAll();
	isRepaint = true;
    }

    public void setActive(String key) {
	if (table.isEmpty()) return;
	TabItem tb;
	if (key == null)
	    tb = (TabItem)getComponent(1);
	else
	    tb = (TabItem)table.get(key);
	selectTab(tb);
    }

    //====== communication method (with TabItem)

    int getCutSize() {return cutSize;}
    double getSpaceParam() {return spaceParam;}
    int getDelta() {return delta;}
    OriginalFont getSelectedFontInfo() {return sFont;  }
    OriginalFont getFontInfo() {return orgFont;   }
    int getStringHeight() {return stringHeight;   }
    int getBottomOffset() {return bottomOffset;   }
    int getTopOffset() {return topOffset;   }
    int getVarticalOffset() {return varticalOffset;   }
    int getHorizontalOffset() {return horizontalOffset;   }

    int getTotalWidth(int ct) {
	return (int)(ct*getSpaceParam())+getHorizontalOffset()*2;
    }
    int getTotalHeight(int ct) {
	return ct+getVarticalOffset()*2+
	    getTopOffset()+getBottomOffset()+delta*2;
    }

    //===== override method area

    public void addNotify() {
	super.addNotify();
	int x=0,y=0;
	if (!table.isEmpty()) {
	    Enumeration e = table.elements();
	    Component tb;
	    Dimension sz;
	    while (e.hasMoreElements()) {
		tb = (Component)e.nextElement();
		sz = tb.getSize();
		x += sz.width;
		y = Math.max(y,sz.height);
	    }
	} else {
	    x = 120;y = 40;
	}
	setSize(x+2,y);
    }

    /**
     * Updates the component. This method is called in
     * response to a call to repaint. You can assume that
     * the background is not cleared.
     * @param g the specified Graphics window
     * @see java.awt.Component#update
     */
    public void update(Graphics g) {
	if (isRepaint) {
	    isRepaint = false;
	}
	super.update(g);
    }

    public void paint(Graphics g) {
	super.paint(g);
	if (table.isEmpty()) {
	    g.drawString("no tab item...",10,5);
	}
    }

    /**
     * jp.gr.java_conf.ccs2.comp.FontInfo stuff
     */
    public Font getFont(int type)  {
	return font1;
    }

    /**
     * jp.gr.java_conf.ccs2.comp.ColorInfo stuff
     */
    public Color getColor(int type)  {
	switch (type) {
	case FORGROUND:
	    return forground;
	case BACKGROUND:
	    return background;
	case LIGHTGRAY:
	    return lightGray;
	case GRAY:
	    return gray;
	case DARKGRAY:
	default:
	    return darkGray;
	}
    }

    //====== private area

    /** change the focus to given TabItem.
      * called by this.setActive,TabItem.<init>
      */
    protected void selectTab(TabItem tb) {
	TabItem pt = (TabItem)table.get(getActive());
	if (tb == pt) {
	    return;
	}
	tb.setActive(true);
	if (pt != null) pt.setActive(false);
	updateImmediate();
	activeTab = tb.getKey();
	sendMessage(tb);
    }

    /** send a CHANGE_TAB message to parent.
      * called by this.selectTab
      */
    protected void sendMessage(TabItem tb) {
	if (parents.isEmpty()) return;
	Enumeration e = parents.elements();
	while(e.hasMoreElements())
	    ((MessageListener)e.nextElement()).message("tab",0,tb.getKey());
    }

    /** called by this.<init>  */
    private void initFont() {
	setFont(stringHeight);
    }

    /** called by this.<init>  */
    private void initLayout() {
	setLayout(new FlowLayout(FlowLayout.LEFT,0,0));
	setBackground(background);
	isRepaint = true;
    }

    /** repaint this component. */
    private void updateImmediate() {
	invalidate();
	doLayout();
	repaint();
    }

    /**
      * It is called automatically by the system the application is started.
      */
    public static void main(String args[]) {
	Frame frame = new Frame("test tab");
	frame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
		System.exit(0);
            }
        });

	Shader sh = new Shader("Tab Component",0xcccccc,0xffffff,0x99bb99,"SansSerif",0,3);
	frame.add(sh,"Center");

	TabCtrl tb = new TabCtrl();
	frame.add(tb,"North");
	tb.addTab("first","first");
	tb.addTab("tab","message");
	tb.addTab("This","this");
	tb.addTab("test TabCtrl.","tab tab");
	tb.addTab("Ok?","this is key");
	tb.setFont(16);
	tb.setColor(Color.blue.darker(),
		    ColorEditor.getMixed(Color.red,Color.white));
	//test
	final TextField tx = new TextField();
	frame.add(tx,"South");
	tb.addMessageListener(new MessageListener() {
	    public void message(String mes,int cd, Object obj) {
 	        tx.setText((String)obj);
	    }
	});
	frame.pack();
	frame.show();
    }
}


/** TabItem manages tab information
  */
    class TabItem extends Canvas {

	RichString contents;
	TabCtrl parent;
	boolean active,isRepaint;
	String key;

	//==== constructor

	TabItem(String title,String key,TabCtrl p) {
	    super();
	    contents = new RichString(title,p.getFontInfo());
	    parent = p;
	    this.key = key;
	    isRepaint = true;
	    addMouseListener(new MouseAdapter() {
		public void mousePressed(MouseEvent e) {
		    parent.selectTab(TabItem.this);
		}
	    });
	}

	//==== access method

	/** update this information to repaint.
	  * called by this.setActive, TabCtrl.setColor.
	  */
	public void updateInfo() {
	    if (isActive()) contents.setFontInfo(parent.getSelectedFontInfo());
	    else contents.setFontInfo(parent.getFontInfo());
	    repaint();
	}

	String getKey() {return key;}

	void setActive(boolean b) {
	    active = b;
	    //isRepaint = true;
	    //(if "isRepaint" is checked, recalculate component size.)
	    updateInfo();
	}
	boolean isActive() {return active;}

	String getTabName() {
	    return contents.getContents();
	}
	void setTabName(String name) {
	    contents.setContents(name);
	    isRepaint = true;
	}

	//====== communication method (with TabCtrl)

	protected Color getDarkGray() {
	    return parent.getColor(TabCtrl.DARKGRAY);
	}
	protected Color getLightGray() {
	    return parent.getColor(TabCtrl.LIGHTGRAY);
	}
	protected Color getTabForground() {
	    return parent.getColor(TabCtrl.FORGROUND);
	}
	protected Color getTabBackground() {
	    return parent.getColor(TabCtrl.BACKGROUND);
	}
	protected Color getGray() {
	    return parent.getColor(TabCtrl.GRAY);
	}

	//==== paint method

	public void addNotify(){
	    super.addNotify();
	    Dimension d = mesureSize();
	    setSize(parent.getTotalWidth(d.width),
		    parent.getTotalHeight(d.height));
	}

	public Dimension getPreferredSize() {
	    return getSize();
	}

	protected Dimension mesureSize() {
	    Dimension d = contents.getSize(this);
	    return d;
	}

	/**
	  * Updates the component. This method is called in
	  * response for a call to repaint. You can assume that
	  * the background is not cleared.
	  * @param g the specified Graphics window
	  * @see java.awt.Component#update
	  */
	public void update(Graphics g) {
	    if (isActive()) contents.setFontInfo(parent.getSelectedFontInfo());
	    else contents.setFontInfo(parent.getFontInfo());
	    paint(g);
	}

	/**
	  * Paints the component.
	  * @param g the specified Graphics window
	  * @see java.awt.Component#paint
	  */
	public void paint(Graphics g) {
	    //check size
	    if (isRepaint) {
		Dimension s = contents.getSize(g);
		setSize(parent.getTotalWidth(s.width),
			parent.getTotalHeight(s.height));
		isRepaint = false;
		parent.doLayout();
	    }
	    Dimension size = getSize();
	    if (size == null) {
		isRepaint = true;
		return;
	    }
	    Color bc = getTabBackground();
	    if (bc != null) {
		//fill background
		g.setColor(bc);
		g.fillRect(0,0,size.width,size.height);
	    }
	    int delta = parent.getDelta();
	    //translate to terminal position
	    size.width --;
	    size.height --;
	    int sx,sy,ex,ey,activeOffset = delta;
	    if (isActive()) {
		activeOffset = 0;
	    }
	    sx = activeOffset;//left-top
	    sy = parent.getTopOffset() + activeOffset;//left-top
	    ex = size.width + activeOffset - delta;//left-bottom
	    ey = size.height - parent.getBottomOffset();//left-bottom

	    //draw contents
	    g.setColor(getTabForground());
	    contents.drawContents(g,sx+parent.getHorizontalOffset(),sy+parent.getVarticalOffset());
	    //border line
	    g.setColor(getGray());
	    //left vartiacal line
	    g.drawLine(sx,sy+parent.getCutSize(),sx,ey);
	    //cut line
	    g.drawLine(sx,sy+parent.getCutSize(),sx+parent.getCutSize(),sy);
	    //top horizontal line
	    g.drawLine(sx+parent.getCutSize(),sy,ex,sy);
	    //right vartical line
	    g.drawLine(ex-1,sy+1,ex-1,ey);
	    //bottom horizotal line
	    g.setColor(getTabBackground());
	    if (!isActive()) {
		g.fillRect(0,ey,ex+delta,ey+parent.getBottomOffset());
	    } else {
		g.drawLine(ex,ey,ex+delta,ey);
		g.fillRect(ex+1,sy+2,delta,ey-3-sy);
	    }

	    //high-light
	    g.setColor(getLightGray());
	    //left vartiacal line
	    g.drawLine(sx+1,sy+parent.getCutSize(),sx+1,ey-1);
	    //cut line
	    g.drawLine(sx+1,sy+parent.getCutSize(),sx+parent.getCutSize(),sy+1);
	    //top horizontal line
	    g.drawLine(sx+parent.getCutSize(),sy+1,ex-2,sy+1);
	    //bottom horizotal line
	    if (!isActive()) {
		g.drawLine(0,ey,ex+delta,ey);
	    } else {
		g.drawLine(ex,ey,ex+delta,ey);
	    }

	    g.setColor(getDarkGray());
	    //right vartical line
	    g.drawLine(ex,sy,ex,ey-1);

	}

    }
    // end of TabItem class
