/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.core;
import java.io.PrintStream;
import java.util.Date;
import java.io.FileWriter;
import java.io.IOException;
import jp.gr.java_conf.ccs2.util.StringUtil;


/** Record and show a message. */
public class MessageMonitor {

    /** debug message */
    public static final int DEBUG = 0;
    /** verbose message */
    public static final int VERBOSE = 1;
    /** normal message */
    public static final int NORMAL = 2;
    /** system meets some error or exception,
	which can be caught. */
    public static final int WARNING = 3;
    /** system meets a fatal error,
        which can not be recovered. */
    public static final int ERROR = 4;

    public static final String FORMAT_DATE = "%date%";
    public static final String FORMAT_LEVEL = "%level%";
    public static final String FORMAT_MESSAGE = "%message%";

    private static final String [] words = {
	"DEBUG","VERBSE","NORMAL","WARNING","ERROR"};

    String filename;
    PrintStream out;
    int logLevel;
    String format = FORMAT_DATE+"|"+FORMAT_LEVEL+"|"+FORMAT_MESSAGE;

    /** construct with given log level and logfile 
	@param filename log filename. if null, no log-file.
	@param logLevel log level.
	@param out console object, such as System.out or System.err.
     */
    public MessageMonitor(String filename,int logLevel,PrintStream out) {
	this.filename = filename;
	this.logLevel = logLevel;
	this.out = out;
    }
    /** construct with NORMAL log level,
	no logfile and STDOUT output. */
    public MessageMonitor() {
	this(null,NORMAL,System.out);
    }
    /** construct with given log level, no logfile
	and STDOUT output.
     */
    public MessageMonitor(int logLevel) {
	this(null,logLevel,System.out);
    }
    /** construct with NORMAL log level,
	given logfile and no output.
     */
    public MessageMonitor(String filename) {
	this(filename,NORMAL,null);
    }

    public int getLogLevel() {return logLevel;}
    public void setLogLevel(int a) {
	if (a >= DEBUG && a <= ERROR) 
	    logLevel = a;
    }
    public PrintStream getOutput() {return out;}
    public void setOutput(PrintStream p) {out = p;}
    public void setFormat(String f) {format = f;}

    public void debug(String message) {
	message(message,DEBUG);
    }
    public void verbose(String message) {
	message(message,VERBOSE);
    }
    public void normal(String message) {
	message(message,NORMAL);
    }
    public void warning(String message) {
	message(message,WARNING);
    }
    public void error(String message) {
	message(message,ERROR);
    }
    protected void message(String message,int level) {
	if (logLevel > level) return;
	String er = formatMessage(message,level);
	if (out != null) out.println(er);
	if (filename != null) {
	    try {
		FileWriter fout = new FileWriter(filename,true);
		fout.write(er + System.getProperty("line.separator"));
		fout.flush();
		fout.close();
	    } catch(IOException e) {
		System.err.println("fatal error : IOException at MessageMonitor.");
		System.err.println(er);
	    }
	}
    }
    String formatMessage(String message,int level) {
	String ret = StringUtil.replace(format,FORMAT_LEVEL,words[level]);
	Date date = new Date();
	ret = StringUtil.replace(ret,FORMAT_DATE,date.toString());
	return StringUtil.replace(ret,FORMAT_MESSAGE,message);
    }
    
    //=====================================

    public static final String defaultName = "ccslog.txt";
    protected static MessageMonitor monitor = 
          new MessageMonitor(defaultName,NORMAL,System.out);

    public static void errorLog(String message) {
	monitor.error(message);
    }
    public static void warningLog(String message) {
	monitor.warning(message);
    }
    public static void messageLog(String message) {
	monitor.normal(message);
    }

}
