/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.io;
import java.io.*;


/** Easy stream reader class. */
public abstract class AbstractReader {

    protected BufferedReader input;

    /** protected constructor 
      * (called by only sub-class)
      */
    protected AbstractReader(BufferedReader in) {
	input = in;
    }

    //======= operation

    /** If file was opened correctly, return true. */
    public boolean isOk() {
	if (input == null) return false;
	try {
	    return input.ready();
	} catch(IOException e) {
	    System.err.println("IOException in AbstractReader.isOk()");
	}
	return false;
    }

    /** return BufferedReader object. */
    public BufferedReader getReader() {return input;}

    /** close the output stream.
      * (you cannot use this object after closed.) 
      */
    public void close() {
	try {
	    input.close();
	} catch(IOException e) {
	    System.err.println(" I/O exception durin closing in AbstractReader.");
	}
    }

    /** check whether this object is available or not
      * @return returns false, if the buffer of this object 
      * becomes empty. Otherwise returns true.
      */
    public boolean isEOF() {
	try {
	    return !input.ready();
	} catch(IOException e) {
	    System.err.println(" I/O exception during EOF check in AbstractReader.");
	}
	return false;
    }

    /** read a line as a string data from this object.
      * @return a string read
      */
    public String read() {
        try {
            return input.readLine();
        } catch (IOException e) {
            System.err.println(" I/O Exception during reading a line in AbstractReader.");
            return "";
        }
    }

    /** read a line as a integer data from this object.
      * @return a read integer
      */
    public int readInteger() { 
        try {
            return Integer.parseInt(read());
        } catch(Exception e) {
            return 0;
        }
    }

    /** read a line as a long integer data from this object.
      * @return a read long integer
      */
    public long readLong() { 
        try {
            return Long.parseLong(read());
        } catch(Exception e) {
            return 0;
        }
    }

    /** read a line as a double explicit flowting point
      *  data from this object.
      * @return a read double
      */
    public double readDouble() {
        try {
            return Double.valueOf(read()).doubleValue();
        } catch(Exception e) {
            return 0;
        }
    }

    /** read a line as a string data from this object.
     * @param def default value
     * @return a string read
     */
    public String read(String def) {
        try {
            return input.readLine();
        } catch (IOException e) {
            System.err.println(" I/O Exception during reading a line in AbstractReader.");
            return def;
        }
    }

    /** read a line as a integer data from this object.
     * @param def default value
      * @return a read integer
      */
    public int readInteger(int def) { 
        try {
            return Integer.parseInt(read());
        } catch(Exception e) {
            return def;
        }
    }

    /** read a line as a long integer data from this object.
     * @param def default value
      * @return a read long integer
      */
    public long readLong(long def) { 
        try {
            return Long.parseLong(read());
        } catch(Exception e) {
            return def;
        }
    }

    /** read a line as a double explicit flowting point
      *  data from this object.
     * @param def default value
      * @return a read double
      */
    public double readDouble(double def) {
        try {
            return Double.valueOf(read()).doubleValue();
        } catch(Exception e) {
            return def;
        }
    }
}
