/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.util;
import java.util.*;
import java.io.Serializable;
import jp.gr.java_conf.ccs2.io.*;


/** Easy access to resource file. */
public class ResourceFile extends ResourceBundle implements Serializable{

    protected Hashtable resourcetable;
    protected String filename;

    //=============================
    // constructor
    //=============================

    /** 
     * Internal constructor
     */
    protected ResourceFile(Hashtable r,String f) {
	resourcetable = r;
	filename = f;
    }

    //=============================
    // override
    //=============================

    /** override */
    public Object handleGetObject(String key) throws MissingResourceException{
	Object obj = resourcetable.get(key);
	if (obj == null) {
	    throw new MissingResourceException("no resource? ["+key+"]",filename,key);
	}
	return obj;
    }

    /** override */
    public Enumeration getKeys() {
	return resourcetable.keys();
    }

    //=============================
    // operation
    //=============================

    /** return true if this object has no elements. */
    public boolean isEmpty() {
	return resourcetable.isEmpty();
    }

    /** return number of elements */
    public int size() {
	return resourcetable.size();
    }

    /** debug dump */
    public String dumpAll() {
	StringBuffer sb = new StringBuffer();
	Enumeration e = getKeys();
	String key;
	if (!e.hasMoreElements()) 
	    return "no contents\n";
	while(e.hasMoreElements()) {
	    key = (String)e.nextElement();
	    sb.append(key).append(" : ").
		append(getString(key)).append("\n");
	}
	return sb.toString();
    }

    /** over-write given table object info this resource file. */
    public void write(Hashtable table) {
	writeFile(filename,table);
    }

    //=============================
    // private area
    //=============================

    //===(read resource file)=======================

    /** read  resource file and make ResourceBundle object.
     * @param file filename of the resource file.
     * @return ResourceBundle object. If the resource file has
     * some problem, this method returns null.
     */
    public static ResourceFile getResourceFile(String file) {
	//read file
	Vector source = readFile(file);
	if (source == null) {
	    System.err.println("file not found...["+file+"]");
	    return null;
	}
	//parse file
	Hashtable table = parse(file,source);
	if (table == null) {
	    System.err.println("parser error...["+file+"]");
	    return null;
	}
	return new ResourceFile(table,file);
    }

    /** [UTILITY]
     * read a file and store lines into a list container 
     */
    public static Vector readFile(String filename) {
	if (filename == null) {
	    return null;
	}
	Vector source = new Vector();
	try {
	    EasyReader reader = new EasyReader(filename);
	    while(!reader.isEOF()) {
		source.addElement(reader.read());
	    }
	} catch(RuntimeException e) {
	    System.err.println(e.getMessage());
	    System.err.println(e.getClass().getName()+
			       " in jp.gr.java_conf.ccs2.jdk11.util.ResourceFile ["+filename+"]");
	    return null;
	}
	return source;
    }

    /** [UTILITY]
     * read a file and store a string object
     */
    public static String file2String(String filename) {
	Vector lines = readFile(filename);
	if (lines == null) {
	    return null;
	}
	StringBuffer in = new StringBuffer();
	for (int i=0;i<lines.size();i++) {
	    String line = (String)lines.elementAt(i);
	    if (line != null)
		in.append( line ).append("\n");
	}
	return in.toString();
    }

    private static Hashtable parse(String file,Vector source) {
	Enumeration e = source.elements();
	Hashtable table = new Hashtable();
	String a = null;
	int line=-1;
	try {
	    while(e.hasMoreElements()) {
		line++;
		a = (String)e.nextElement();
		if (a == null) continue;//blank line
		if (a.length() == 0) continue;//blank line
		if (a.charAt(0)=='#') continue;//comment line
		int ps = a.indexOf("=");
		if (ps == -1) {
		    System.out.println(file+" : syntax error in "+line);
		    continue;//error
		}
		table.put(a.substring(0,ps),a.substring(ps+1));
	    }
	    return table;
	} catch(Exception ex) {
	    System.err.println("ERROR : "+a +" : in "+line);
	    return null;
	}
    }

    /** utility method. write given table object into given filename.
     * @param file filename
     * @param table Hashtable object
     */
    public static void writeFile(String file,Hashtable table) {
	if (file == null || table==null) return;
	
	EasyWriter out = new EasyWriter(file);
	Enumeration e = table.keys();
	while(e.hasMoreElements()) {
	    String k = (String)e.nextElement();
	    out.println(k+"="+(String)table.get(k));
	}
	out.flush();
	out.close();
	System.out.println("wrote...");
    }
}
