/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.comp;
import java.awt.*;
import java.awt.image.*;
import java.net.*;


/** paint background image */
public class ImagePainter extends RectPainter {

    public static final int CENTER  = 0;
    public static final int FILL    = 1;
    public static final int FIT     = 2;
    public static final int LEFTTOP = 3;

    protected Image image;
    protected int type = CENTER;
    protected ImageObserver parent;

    //================================
    // constructor
    //================================


    /** constructor
      *@param i source image object
      *@param t paint type (CENTER,FILL,FIT,LEFTTOP)
      *@param p ImageObserver object
      */
    public ImagePainter(Image i,int t,ImageObserver p) {
	image = i;
	type = t;
	parent = p;
    }

    /** constructor
      *@param path source image path or URL
      *@param t paint type (CENTER,FILL,FIT,LEFTTOP)
      *@param p ImageObserver object
      */
    public ImagePainter(String path,int t,ImageObserver p) {
	loadImage(path);
	type = t;
	parent = p;
    }

    /** constructor (use CENTER paint mode)
      *@param i source image object
      *@param p ImageObserver object
      */
    public ImagePainter(Image i,ImageObserver p) {
	this(i,CENTER,p);
    }

    /** constructor (use CENTER paint mode)
      *@param i source image path or URL
      *@param p ImageObserver object
      */
    public ImagePainter(String path,ImageObserver p) {
	this(path,CENTER,p);
    }

    private void loadImage(String path) {
	try {
	    image = Toolkit.getDefaultToolkit().getImage(path);
 	} catch(Exception e) {
	    System.err.println(e.getClass().getName()+" : "+e.getMessage());
	    System.err.println("Cannot load image. ["+path+"]");
	    //e.printStackTrace();
	}
    }

    //================================
    // access
    //================================

    public void setImage(String path) { loadImage(path);}
    public void setImage(Image i) { image = i; }
    public Image getImage() {return image;}
    public void setType(int i) { type = i; }
    public int getType() {return type;}

    //================================
    // operation
    //================================

    public void filter(Color c) {
	ImageFilter f = new ColoredFilter(c);
	ImageProducer pro = new FilteredImageSource(image.getSource(),f);
	image = Toolkit.getDefaultToolkit().createImage(pro);

    }
    
    //================================
    // private
    //================================

    protected void work(Graphics g,Rectangle r) {
	int x=r.x,y=r.y;
	int w = image.getWidth(parent);
	int h = image.getHeight(parent);
	switch (type) {
	case FILL:
	    if (w <= 0 || h <= 0) return;
	    while(y < (r.height+r.y)) {
		x = r.x;
		while(x < (r.width+r.x)) {
		    g.drawImage(image,x,y,parent);
		    x += w;
		}
		y += h;
	    }
	    break;
	case FIT:
	    g.drawImage(image,r.x,r.y,r.width,r.height,parent);
	    break;
	case LEFTTOP:
	    g.drawImage(image,r.x,r.y,parent);
	    break;
	case CENTER:
	default:
	    x = (r.width-w)>>1;
	    y = (r.height-h)>>1;
	    g.drawImage(image,r.x+x,r.y+y,parent);
	    break;
	}
    }

    class ColoredFilter extends RGBImageFilter {
	int rr,gg,bb;
	public ColoredFilter(Color c) {
	    // The filter's operation does not depend on the
	    // pixel's location, so IndexColorModels can be
	    // filtered directly.
	    canFilterIndexColorModel = true;
	    rr = c.getRed();
	    gg = c.getGreen();
	    bb = c.getBlue();
	}
	
	public int filterRGB(int x, int y, int rgb) {
	    int a = (rgb & 0xff000000);
	    int g = (rgb & 0x00ff00) >> 8;
	    int r = (rgb & 0xff0000) >> 16;
	    int b = (rgb & 0x0000ff);
	    g = (gg+g)>>1;
	    r = (rr+r)>>1;
	    b = (b+bb)>>1;
	    return (r<<16)+(g<<8)+b+a;
	}
    }

}

