/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.comp;
import java.awt.*;


/** string object with a font and a color information.
    <pre>
    +----------------------------------------+
    |   vartical offset | (default 3dot)     |
    |  +----------------------------------+  |
    |  |                                  |  |
    |  |   contents (foreground)          |  |
    |  |                                  |  |
    |  |           (background)           |  |
    | =+----------------------------------+  |
    | holizontal offset (default 3dot)       |
    +----------------------------------------+
    </pre>    
 */

public class RichString {
    protected FontInfo orgFont;
    protected int fontNumber=0;
    protected String contents;
    protected int voff = 3; // vartical offset
    protected int hoff = 3; // holizontal offset
    private int ac,dc;             //used in drawContents
    private boolean changed = true;//used in getSize
    private Dimension currentSize;

    //===============
    // constructor
    //===============

    /** default constructor (easy access) <br>
      * (null string,black color,transparent background,<br>
      * SansSerif,PLAIN style,11 point)
      */
    public RichString() {
	this("",Color.black,null,new Font("SansSerif",Font.PLAIN,12) );
    }

    /** 
	@param c contents string
	@param f foreground color
	@param b background color
	@param a font
     */
    public RichString(String c,Color f,Color b,Font a) {
	contents = c;
	orgFont = new OriginalFont(f,b,a);
    }

    /** 
	@param c contents string
	@param orf FontInfo object
     */
    public RichString(String c,FontInfo orf) {
	this(c,orf,0);
    }

    /** 
	@param c contents string
	@param orf FontInfo object
	@param num an index number on FontInfo object
     */
    public RichString(String c,FontInfo orf,int num) {
	contents = c;
	orgFont = orf;
	fontNumber = num;
    }

    //===============
    // access method
    //===============

    /** set FontInfo object (you should repaint to update this change) */
    void setFontInfo(FontInfo f) {
	orgFont = f;
	changed = true;
    }

    /** set font number (you should repaint to update this change) */
    void setFontNumber(int f) {
	fontNumber = f;
	changed = true;
    }

    /** access to contents string */
    public void setContents(String s) { 
	contents = s; 
	changed = true;
    }

    /** get contents string */
    public String getContents() { return contents;}

    /** access to FontInfo. */
    public FontInfo getFontInfo() {return orgFont;}

    /** get forground color (easy access) */
    public Color getForeground() { 
	return orgFont.getColor(OriginalFont.FORGROUND);
    }

    /** get background color (easy access) */
    public Color getBackground() { 
	return orgFont.getColor(OriginalFont.BACKGROUND);
    }

    /** get font (easy access) */
    public Font getFont() { 
	return orgFont.getFont(fontNumber);
    }

    /** access to offset parameters */
    public int getVarticalOffset() {return voff;}
    /** access to offset parameters */
    public int getHolizontalOffset() {return hoff;}
    /** access to offset parameters */
    public void setVarticalOffset(int v) {voff = v;}
    /** access to offset parameters */
    public void setHolizontalOffset(int h) {hoff = h;}

    //===============
    // operation
    //===============

    /** return demension size */
    public Dimension getSize(Component c) {
	if (changed == true) 
	    mesureSize(c.getFontMetrics(getFont()));
	return currentSize;
    }

    /** return demension size */
    public Dimension getSize(Graphics g) {
	if (changed == true) 
	    mesureSize(g.getFontMetrics(getFont()));
	return currentSize;
    }

    /** mesure String size */
    protected void mesureSize(FontMetrics fm) {
	currentSize = new Dimension();
	if (fm != null) {
	    currentSize.height = fm.getHeight()+voff*2;
	    currentSize.width = fm.stringWidth(contents)+hoff*2;
	    ac = fm.getAscent();
	    dc = fm.getDescent();
	    changed = false;
	} else {
	    System.out.println("Null FontMetrics");
	    //approximate size (not equal!)
	    currentSize.height = getFont().getSize()+hoff*2;
	    currentSize.width = getFont().getSize()*(contents.length());
	}
    }

    //===============
    // private area
    //===============

    /** draw method
      * @param g graphics context
      * @param x left position
      * @param y upper position
      * @return size of this string label
      */
    public Dimension drawContents(Graphics g,int x,int y) {
	Dimension d = getSize(g);
	Color bc = getBackground();
	if (bc != null) {
	    g.setColor(bc);
	    g.fillRect(x,y,d.width,d.height);
	}
	y += ac + voff;	x += hoff;
	g.setColor(getForeground());
	g.setFont(getFont());
	g.drawString(contents,x,y);
	return d;
    }

}

