/* CCS, Class Collection by Sakurai
 *  Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.io;
import java.io.*;
import java.util.*;


/** ClassFileLoader is a custom class loader, which reads
  * a class file and builds Class object.
  * 
  */
public class ClassFileLoader extends ClassLoader {

    protected Hashtable classes = new Hashtable();
    protected String path;

    /** default constructor
      * the base class path is set current directory.
      */
    public ClassFileLoader() {
	path = "";
    }

    /** standerd constructor
      * @param p base class path
      */
    public ClassFileLoader(String p) {
	if (p.length()!=0) {
	    if (!p.endsWith("/"))
		p += "/";
	}
	path = p;
    }

    /** public operation method.
      * @param name abstract class name (not include ".class" sufix)
      * @param r resolve switch
      */
    public Class load(String name,boolean r) throws ClassNotFoundException {
	    return loadClass(name,r);
    }

    /** over-ride method
      * @param name abstract class name (not include ".class" sufix)
      * @param r resolve switch
      * @return Class object which you want
      */
    protected Class loadClass(String name,boolean r) throws ClassNotFoundException {
	Class cl = (Class)classes.get(name);
	if (cl == null) {
	    try { 
		return findSystemClass(name);
	    } catch (Exception e) { }
	    cl = buildClass(name);
	}
	if (cl == null) {
	    throw new ClassNotFoundException(name);
	}
	if (r) {
	    resolveClass(cl);
	}

	return cl;
    }

    /** get Class object by name
      * @param name abstract class name
      * @return Class object which you want
      */
    private Class buildClass(String name) {
	try {
	    BufferedInputStream in = new BufferedInputStream(
				     new FileInputStream(path+name+".class") );
	    return loadClass(in);
	} catch(Exception e) {
	    System.out.println("class file not found : "+name);
	}
	return null;
    }

    /** get Class object by InputStream
      * @param in InputStream object which opens the class file
      * @return Class object which you want
      */
    private Class loadClass(InputStream in) throws IOException { 
	try {
	    try {
		byte [] data = new byte[4096];
		int total = 0,n;
		
		while( (n=in.read(data,total,data.length-total)) >= 0) {
		    if ( (total+=n) == data.length) {
			byte [] newdata = new byte[total*2];
			System.arraycopy(data,0,newdata,0,total);
			data = newdata;
		    }
		}
		return defineClass(null,data,0,total);
	    } finally {
		if (in != null) {
		    in.close();
		}
	    }
	} catch(Exception e) {
	    e.printStackTrace();
	    throw new IOException("class not loaded");
	}
    }

    // test method
    public static void main(String [] args) {
	ConsoleReader cin = new ConsoleReader();
	String filename="";
	if (args.length==0) {
	    System.out.println("input class name");
	    filename = cin.read();
	} else {
	    filename = args[0];
	}

	try {
	    ClassFileLoader god = new ClassFileLoader("");
	    Class guy = god.load(filename,true);
	    System.out.println("class ["+guy.getName()+"] loaded.");
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }
}

