/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax;
import java.util.*;
import java.io.*;
import jp.gr.java_conf.ccs2.util.StringUtil;


public class ModuleModel implements ElementOperation,Serializable {

	static final long serialVersionUID = 2505549755525109268L;

	private List attributePools = new ArrayList();
	private List hedgeRules = new ArrayList();
	private List objects = new ArrayList();
	private List includedModules = new ArrayList();

	private String moduleFilename;
	private String targetNameSpace;
	private String moduleComment;
	private String moduleVersion;
	private String relaxCoreVersion = "1.0";

	private transient boolean modifiedFlag = false;

	public ModuleModel(String comment,String ns,String ver) {
		setModuleVersion(ver);
		setModuleComment(comment);
		setTargetNameSpace(ns);
	}

	public ModuleModel getParentModule() {return this;}

	public void addRootObject(ObjectModel obj) {
		if (obj == null) return;
		objects.add(obj);
	}

	public void removeRootObject(ObjectModel obj) {
		if (obj == null) return;
		objects.remove(obj);
	}

	public ObjectModel [] getRootObjects() {
		ObjectModel [] om = new ObjectModel[objects.size()];
		return (ObjectModel[])objects.toArray(om);
	}

	public ObjectModel getObjectByLabel(String label) {
		ObjectModel [] models = getAllObjects();
		for (int i=0;i<models.length;i++) {
			if (models[i].getLabel().equals(label)) return models[i];
		}
		return null;
	}

	public ObjectModel [] getObjects() {
		return getObjects(false);
	}

	public ObjectModel [] getExportableObjects() {
		return getObjects(true);
	}

	public ObjectModel [] getExportableAllObjects() {
		ElementCollectVisitor ev = new ElementCollectVisitor(true);
		VisitorCenter.traverseAll(this,ev);
		return ev.getElements();
	}

	private ObjectModel [] getObjects(boolean t) {
		ElementCollectVisitor ev = new ElementCollectVisitor(t);
		VisitorCenter.traverseAllInsideObjects(this,ev);
		return ev.getElements();
	}

	public ObjectModel [] getAllObjects() {
		ElementCollectVisitor ev = new ElementCollectVisitor();
		VisitorCenter.traverseAllAroundModules(this,ev);
		return ev.getElements();
	}

	public AttributePoolModel getAttributePoolByRole(String role) {
		AttributePoolModel [] models = getAllAttributePools();
		for (int i=0;i<models.length;i++) {
			if (models[i].getRole().equals(role)) return models[i];
		}
		return null;
	}

	public void addAttributePool(AttributePoolModel obj) {
		if (obj == null) return;
		attributePools.add(obj);
	}

	public void removeAttributePool(AttributePoolModel obj) {
		if (obj == null) return;
		attributePools.remove(obj);
	}

	public AttributePoolModel [] getAttributePools() {
		return getAttributePools(null);
	}

	public AttributePoolModel [] getAttributePools(AttributePoolModel exclude) {
		List tList = new ArrayList(attributePools);
		if (exclude != null) {
			tList.remove(exclude);
		}
		AttributePoolModel [] om = new AttributePoolModel[tList.size()];
		return (AttributePoolModel[])tList.toArray(om);
	}

	public AttributePoolModel [] getAllAttributePools() {
		return getAllAttributePools(null);
	}

	public AttributePoolModel [] getAllAttributePools(AttributePoolModel exclude) {
		List list = new ArrayList(Arrays.asList(getAttributePools(exclude)));
		ModuleModel [] ms = getIncludeModules();
		for (int i=0;i<ms.length;i++) {
			list.addAll(Arrays.asList(ms[i].getAttributePools(exclude)));
		}
		AttributePoolModel [] ret = new AttributePoolModel[list.size()];
		return (AttributePoolModel[])list.toArray(ret);
	}

	public HedgeRuleElementModel getHedgeRuleByLabel(String label) {
		HedgeRuleElementModel [] models = getAllHedgeRules();
		for (int i=0;i<models.length;i++) {
			if (models[i].getLabel().equals(label)) return models[i];
		}
		return null;
	}

	public void addHedgeRule(HedgeRuleElementModel obj) {
		if (obj == null) return;
		hedgeRules.add(obj);
	}

	public void removeHedgeRule(HedgeRuleElementModel obj) {
		if (obj == null) return;
		hedgeRules.remove(obj);
	}

	public HedgeRuleElementModel [] getHedgeRules() {
		return getHedgeRules(null);
	}

	public HedgeRuleElementModel [] getHedgeRules(HedgeRuleElementModel exclude) {
		List tList =  new ArrayList(hedgeRules);
		if (exclude != null) {
			tList.remove(exclude);
		}
		HedgeRuleElementModel [] om = new HedgeRuleElementModel[tList.size()];
		return (HedgeRuleElementModel[])tList.toArray(om);
	}

	public HedgeRuleElementModel [] getAllHedgeRules() {
		return getAllHedgeRules(null);
	}
	public HedgeRuleElementModel [] getAllHedgeRules(HedgeRuleElementModel exclude) {
		List list = new ArrayList(Arrays.asList(getHedgeRules(exclude)));
		ModuleModel [] ms = getIncludeModules();
		for (int i=0;i<ms.length;i++) {
			list.addAll(Arrays.asList(ms[i].getHedgeRules(exclude)));
		}
		HedgeRuleElementModel [] ret = new HedgeRuleElementModel[list.size()];
		return (HedgeRuleElementModel[])list.toArray(ret);
	}

	public void addIncludeModule(ModuleModel module) {
		if (module == null) return;
		includedModules.add(module);
	}

	public void removeIncludeModule(ModuleModel module) {
		if (module == null) return;
		includedModules.remove(module);
	}

	public ModuleModel [] getIncludeModules() {
		ModuleModel [] om = new ModuleModel[includedModules.size()];
		return (ModuleModel[])includedModules.toArray(om);
	}
	
	public String getRelaxCoreVersion() {
		return relaxCoreVersion;
	}
	public void setRelaxCoreVersion(String	v) {
		this.relaxCoreVersion = v;
	}
	
	public String getModuleVersion() {
		return moduleVersion;
	}
	public void setModuleVersion(String	 v) {
		this.moduleVersion = v;
	}
	public String getTargetNameSpace() {
		return targetNameSpace;
	}
	public void setTargetNameSpace(String  v) {
		this.targetNameSpace = v;
	}
	public String getModuleComment() {
		return moduleComment;
	}
	public void setModuleComment(String	 v) {
		this.moduleComment = v;
	}
	public String getModuleName() {
		if (StringUtil.isNull(moduleFilename))
			return MainApp.getResString("noFilename");
		File file = new File(moduleFilename);
		return file.getName();
	}
	public String getModuleFilename() {
		return moduleFilename;
	}
	public void setModuleFilename(String  v) {
		this.moduleFilename = v;
	}

	// document operation

	public void modified() {setModified(true);}
	protected void clearModification() {setModified(false);}
	protected void setModified(boolean b) {
		modifiedFlag = b;
	}
	public boolean isModified() {
		return modifiedFlag;
	}

	// Element operation stuff

	public boolean canAddElement() {return true;}
	public void addOpElements(ElementOperation [] args) {
		if (args == null ) return;
		for (int i=0;i<args.length;i++) {
			if (args[i] instanceof ObjectModel) {
				addRootObject((ObjectModel)args[i]);
			}
		}
	}

	public ElementOperation [] getOpElements() {
		if (objects.size()==0) return null;
		ElementOperation [] ar = new ElementOperation[objects.size()];
		return (ElementOperation[]) objects.toArray(ar);
	}

	public void removeOpElement(ElementOperation arg) {
		if (arg == null ) return;
		if (arg instanceof ObjectModel) {
			removeRootObject((ObjectModel)arg);
		}
	}

	public ElementOperation getOpCopy() {
		try {
			MainApp.getAppContext().getMonitor().debug("Module copied.");
			return relax2module(module2relax(this),getModuleFilename());
		} catch (MkRelaxException e) {
			MainApp.getAppContext().getMonitor().recordStackTrace(e);
		}
		return null;
	}

	public int getOpElementLevel() {
		throw new InternalError("Module is top level component. Can not do this operation.(bug)");
	}

	public boolean acceptOpElement(ElementOperation e) {
		if (e == null) return false;
		if (e.getOpElementLevel() == OBJECT) return true;
		return false;
	}

	public String getRelax() {
		return module2relax(this);
	}

	// static utility

	public static String module2relax(ModuleModel model,String encoding,
									  boolean forceCorrect) {
		RelaxGenVisitor gen = new RelaxGenVisitor(encoding);
		gen.setForceCorrect(forceCorrect);
		VisitorCenter.traverseAllInsideObjects(model,gen);
		return gen.getSource();
	}

	public static String module2relax(ModuleModel model) {
		return module2relax(model,null,false);
	}

	public static ModuleModel relax2module(String input,String filename) throws MkRelaxException {
		return RelaxReader.getModuleFromString(input,filename);
	}

	public static ModuleModel createNewModule() {
		ModuleModel m = new ModuleModel("","","1.0");
		m.setModuleFilename("");
		return m;
	}

	public static void saveModuleWithRelax(ModuleModel model,String filename) throws IOException{
		if (StringUtil.isNull(filename))
			throw new IOException(MainApp.getResString("SaveNullFilename"));
		String encoding = MainApp.getAppContext().
			getConfig().getOptionString("encoding");
		PrintWriter out = null;
		try {
			out = new PrintWriter
				(new OutputStreamWriter
					(new FileOutputStream(filename),encoding));
			out.println(module2relax((ModuleModel)model.getOpCopy(),
									 encoding,true));
			out.flush();
			model.clearModification();
		} finally {
			if (out != null) 
				out.close();
		}
	}

	public static ModuleModel loadModuleFromRelax(String filename) throws IOException {
		try {
			File file = new File(filename);
			if (!file.exists()) {
				throw new IOException(MainApp.getResString("LoadModuleFileNotFound{0}",filename));
			}
			ModuleModel module = RelaxReader.getModuleFromFile(filename);
			module.clearModification();
			return module;
		} catch (RuntimeException re) {
			re.printStackTrace();
			throw new IOException(MainApp.getResString("LoadModuleGotRuntimeException")+" : "+re.getMessage());
		} catch (MkRelaxException re) {
			throw new IOException(re.getMessage());
		}
	}
	
}


