/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax;

import java.io.IOException;
import java.io.StringWriter;
import java.util.*;
import javax.xml.parsers.*;
import org.apache.xml.serialize.*;
import org.w3c.dom.*;
import jp.gr.java_conf.ccs2.util.StringUtil;

public class RelaxGenVisitor extends VisitorClass {
	
	//Temporary Document root
	protected static Document newDocument() {
		try {
			DocumentBuilderFactory factory
				= DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			return builder.newDocument();
		} catch (ParserConfigurationException e) {
			throw (new InternalError(e.getMessage()));
		}
	}

	private Document document = newDocument();
	private ModuleModel module;
	private String encoding;
	private Element root;
	private RefElementModel reference;//temporary object 
	private Stack parentStack = new Stack();
	private List wroteObjectList = new ArrayList();
	private boolean forceCorrect = false;

	private Element curParent() {
		return (Element)parentStack.peek();
	}

	public RelaxGenVisitor() {
	}

	public RelaxGenVisitor(String encoding) {
		this.encoding = encoding;
	}

	public void setForceCorrect(boolean f) {
		forceCorrect = f;
	}

	public int getAllowedNestedLoop() {return 0;}

	public boolean isOnlyFirstContact() {
		return true;
	}

	public String getSource() {
		StringWriter out = new StringWriter();
		OutputFormat format = new OutputFormat(document);
		if (!StringUtil.isNull(encoding))
			format.setEncoding(encoding);
		format.setIndenting(true);
		XMLSerializer ser = new XMLSerializer(out,format);
		try {
			ser.serialize(document);
		} catch (IOException e) {
			throw new RuntimeException("IOException? : "+e.getMessage());
		}
		return out.toString();
	}
	
	public void start(ModuleModel m) {
		module = m;
		root = document.createElement("module");
		root.setAttribute("moduleVersion",m.getModuleVersion());
		root.setAttribute("relaxCoreVersion",m.getRelaxCoreVersion());
		root.setAttribute("targetNamespace",m.getTargetNameSpace());
		root.setAttribute("xmlns","http://www.xml.gr.jp/xmlns/relaxCore");
		document.appendChild(root);
		Element an = appendAnnotationE(root,module.getModuleComment());
		if (an != null) {
			Element nameNode = document.createElement("appinfo");
			nameNode.appendChild
				(document.createTextNode("ModulePath:"+module.getModuleFilename()));
			an.appendChild(nameNode);
		}
	
		ModuleModel [] includedModule = module.getIncludeModules();
		for (int i=0;i<includedModule.length;i++) {
			Element include = document.createElement("include");
			include.setAttribute("moduleLocation",
								 includedModule[i].getModuleFilename());
			root.appendChild(include);
		}
	}

	protected Element appendAnnotation(Element element,String comment) {
		if (StringUtil.isNull(comment)) return null;
		return appendAnnotationE(element,comment);
	}

	protected Element appendAnnotationE(Element element,String comment) {
		if (!MainApp.getAppContext().getConfig().getOption("annotation")) return null;
		Element an = document.createElement("annotation");
		Element dd = document.createElement("documentation");
		dd.appendChild( document.createTextNode( comment ));
		an.appendChild( dd );
		element.appendChild( an );
		return an;
	}

	public void end(ModuleModel m) {
		//add attribute pool
		AttributePoolModel [] attrs = module.getAttributePools();
		for(int i=0;i<attrs.length;i++) {
			Element attElement = null;
			if (attrs[i] instanceof AttributePoolModel) {
				AttributePoolModel ct = (AttributePoolModel)attrs[i];
				attElement = document.createElement("attPool");
				appendAnnotation(attElement,ct.getComment());
				attElement.setAttribute("role",ct.getRole());
				AttributeModel [] ats = ct.getAttributes();
				for (int j=0;j<ats.length;j++) {
					appendAttribute(document,attElement,ats[j]);
				}
			} else {
				throw new InternalError("No such attribute. ["+
										attrs[i].getClass().getName()+
										" : "+attrs[i]+"]");
			}
			root.appendChild(attElement);
		}
	}
	
	public void root(ObjectModel obj) { 
		ObjectModel [] objs = module.getExportableObjects();
		Element intf = document.createElement("interface");
		for (int i=0;i<objs.length;i++) {
			Element exp = document.createElement("export");
			exp.setAttribute("label",objs[i].getLabel());
			intf.appendChild(exp);
		}
		root.appendChild(intf);
		parentStack.push(root);
	}

	public void objectEnter(ObjectModel obj,int loop) {
		if (wroteObjectList.contains(obj)) {
			throw new InternalError("Second Encounterd:"+obj);
		}
		if (obj.getType() == DataType.RLX_ELEMENT && 
			obj.getElement() == null && forceCorrect) {
			obj.setType(DataType.DEFAULT_TYPE);
		}
		if (obj.isLabelEqTag() && 
			obj.getType() != DataType.RLX_ELEMENT &&
			obj.getElement() == null &&	
			reference != null &&
			obj.getAttributeNum() == 0 && 
			ElementUtil.getReferencesToMe(module,obj).length == 1) {
			//output as <element>
			Element element = document.createElement("element");
			appendAnnotation(element,obj.getComment());
			element.setAttribute("name",obj.getTagName());
			element.setAttribute("type",obj.getType().getTypeName());
			if (reference.getOccurs() != reference.OCCURS_ONE) {
				element.setAttribute("occurs",reference.getOccurSymbol());
			}
			curParent().appendChild(element);
			parentStack.push(element);
		} else {
			//output as <elementRule> and <tag>
			if (reference != null) {
				// make <ref>
				Element ref = document.createElement("ref");
				ref.setAttribute("label",reference.getAt().getLabel());
				setupOccus(ref,reference);
				curParent().appendChild(ref);
			}
			// make <elementRule> <tag>
			Element element = document.createElement("elementRule");
			appendAnnotation(element,obj.getComment());
			Element tag = document.createElement("tag");
			if (!obj.isLabelEqTag())
				tag.setAttribute("name",obj.getTagName());
			AttributeModel [] attrs = obj.getAttributes();
			for(int i=0;i<attrs.length;i++) {
				appendAttribute(document,tag,attrs[i]);
			}
			element.appendChild(tag);
			element.setAttribute("label",obj.getLabel());
			if (obj.getType() != DataType.RLX_ELEMENT) {
				element.setAttribute("type",obj.getType().getTypeName());
			}
			root.appendChild(element);
			parentStack.push(element);
		}
		reference = null;
	}

	public void objectExit(ObjectModel obj) {
		if (wroteObjectList.contains(obj)) return;
		parentStack.pop();
		wroteObjectList.add(obj);
	}

	public void hedgeRuleEnter(HedgeRuleElementModel obj) {
		Element element = document.createElement("hedgeRule");
		element.setAttribute("label",obj.getLabel());
		appendAnnotation(element,obj.getComment());
		root.appendChild(element);
		parentStack.push(element);
	}

	public void hedgeRuleExit(HedgeRuleElementModel obj) {
		parentStack.pop();
	}

	void appendAttribute(Document doc,Element element,AttributeModel attribute) {
		Element attElement = null;
		if (attribute instanceof ConcreteAttributeModel) {
			ConcreteAttributeModel ct = (ConcreteAttributeModel)attribute;
			attElement = document.createElement("attribute");
			appendAnnotation(attElement,ct.getComment());
			attElement.setAttribute("name",ct.getName());
			attElement.setAttribute("type",ct.getType().getTypeName());
			if (ct.isRequired()) {
				attElement.setAttribute("required","true");
			}
		} else if (attribute instanceof AttributeRefModel) {
			AttributeRefModel ar = (AttributeRefModel)attribute;
			attElement = document.createElement("ref");
			attElement.setAttribute("role",ar.getAt().getRole());
		} else {
			throw new InternalError("No such attribute. ["+
									attribute.getClass().getName()+
									" : "+attribute+"]");
		}
		element.appendChild(attElement);
	}

	public void mixedEnter(MixedModel obj) {
		Element e = document.createElement("mixed");
		curParent().appendChild(e);
		parentStack.push(e);
	}

	public void mixedExit(MixedModel obj) {
		parentStack.pop();
	}

	public void none(NoneElementModel obj) {
		curParent().appendChild(document.createElement("none"));
	}
	public void empty(EmptyElementModel obj) {
		curParent().appendChild(document.createElement("empty"));
	}

	public void choiceEnter(ChoiceElementModel obj) {
		Element choice = document.createElement("choice");
		setupOccus(choice,obj);
		curParent().appendChild(choice);
		parentStack.push(choice);
	}
	public void choiceExit(ChoiceElementModel obj) {
		parentStack.pop();
	}

	public void sequenceEnter(SequenceElementModel obj) {
		Element sequence = document.createElement("sequence");
		setupOccus(sequence,obj);
		curParent().appendChild(sequence);
		parentStack.push(sequence);
	}
	public void sequenceExit(SequenceElementModel obj) {
		parentStack.pop();
	}

	public void hedgeRef(HedgeRefElementModel obj) {
		Element hedgeRef = document.createElement("hedgeRef");
		hedgeRef.setAttribute("label",obj.getAt().getLabel());
		setupOccus(hedgeRef,obj);
		curParent().appendChild(hedgeRef);
	}

	public void ref(RefElementModel ref) {
		if (ref.getAt() != null &&
			(ref.getAt().getParentModule() != module ||
			 ElementUtil.getReferencesToMe(module,ref.getAt()).length > 1)) {
			// make <ref>
			Element refTag = document.createElement("ref");
			refTag.setAttribute("label",ref.getAt().getLabel());
			setupOccus(refTag,ref);
			curParent().appendChild(refTag);
			return;
		}
		reference = ref;
	}

	private void setupOccus(Element element,AbstractElementModel obj) {
		if (obj.getOccurs() != obj.OCCURS_ONE) {
			element.setAttribute("occurs",obj.getOccurSymbol());
		}
	}
}
