/* MkRelax, Visual Relax Editor
 *  Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax;

import java.io.IOException;
import java.io.StringWriter;
import java.io.StringReader;
import java.io.File;
import java.util.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;
import org.apache.xml.serialize.*;
import jp.gr.java_conf.ccs2.util.StringUtil;
import jp.gr.xml.relax.sax.RELAXEntityResolver;
import java.text.MessageFormat;

public class RelaxReader {

    ModuleModel moduleModel,oldModule;
    List reloadEntries;

    HashMap tagNodeTable = new HashMap();

    HashMap ruleTable = new HashMap();
    HashMap tagTable = new HashMap();
    HashMap attPoolTable = new HashMap();
    HashMap hedgeRuleTable = new HashMap();

    List refList = new ArrayList();
    String curPath;

    Stack pathStack = new Stack();
    
    protected RelaxReader() {
    }

    public static ModuleModel getModuleWithReloading(ModuleModel oldModule,List reloadEntries) throws MkRelaxException {
	RelaxReader r = new RelaxReader();
	r.setupModuleWithReloading(oldModule,reloadEntries);
	return r.getModule();
    }

    //reloadEntries == null -> reload all modules
    protected void setupModuleWithReloading(ModuleModel old,List reloadEntries) throws MkRelaxException {
	oldModule = old;
	setCurrentPath(oldModule.getModuleFilename());
	this.reloadEntries = reloadEntries;
	setupModuleCommon(readDocumentFromString(old.getRelax()));
    }

    public static ModuleModel getModuleFromFile(String filename) throws MkRelaxException {
	RelaxReader r = new RelaxReader();
	r.setupModuleFromFile(filename);
	return r.getModule();
    }

    protected void setupModuleFromFile(String filename) throws MkRelaxException {
	setCurrentPath(filename);
	setupModuleCommon(readDocumentFromFile(filename));
    }

    public static ModuleModel getModuleFromString(String input,String filename) throws MkRelaxException {
	RelaxReader r = new RelaxReader();
	r.setupModuleFromString(input,filename);
	return r.getModule();
    }

    public void setupModuleFromString(String input,String filename) throws MkRelaxException {
	setCurrentPath(filename);
	setupModuleCommon(readDocumentFromString(input));
    }

    protected void setupModuleCommon(Document document) throws MkRelaxException {
	try {
	    pathStack.push("bottom");
	    expandRelax(document);
	    buildModule(document);
	    if (pathStack.size() != 1) 
		throw new InternalError
		    ("Wrong number of stacked objects(bug) : "+
		     pathStack.size());
	} catch (RuntimeException e) {
	    throwException(format("RelaxSyntaxError",e.getMessage()));
	}
    }

    protected ModuleModel getModule() {
	MainApp.getAppContext().getMonitor().debug("READER:"+getDumpModule(moduleModel));
	return moduleModel;
    }

    void buildModule(Document document) throws MkRelaxException {
	Element moduleElement = document.getDocumentElement();
	setupModule(moduleElement);
	prepareTables(moduleElement);
	setupInterface(moduleElement);
	includeModules(moduleElement);
	setupElementRules(moduleElement);
	setupHedgeRules(moduleElement);
	setupAttPools(moduleElement);
	findRoots();
    }

    void includeModules(Element moduleNode) throws MkRelaxException {
	Element [] includeNodes = getElementsByTagName(moduleNode,"include");
	for (int i=0;i<includeNodes.length;i++) {
	    String location = includeNodes[i].getAttribute("moduleLocation");
	    if (StringUtil.isNull(location))
		throwException(format("ModuleLocationIsNull{0}",
				      includeNodes[i] ));
	    if (oldModule != null) {
		ModuleModel mod = getOldModuleByLocation(location);
		if (mod == null) {
		    throwException
			(format("Module object has gone! : {0}",location));
		}
		if (reloadEntries != null && !reloadEntries.contains(mod)) {
		    moduleModel.addIncludeModule
			(getModuleWithReloading(mod,reloadEntries));
		    MainApp.getAppContext().getMonitor().debug("READER : rebuild module ["+location+"]");
		    continue;
		}
	    }
	    try {
		moduleModel.addIncludeModule
		    (ModuleModel.loadModuleFromRelax(transCanonicalPath(location)));
		MainApp.getAppContext().getMonitor().debug("READER : loaded from ["+location+"]");
	    } catch (IOException e) {
		throwException(e.getMessage());
	    }
	}
    }

    ModuleModel getOldModuleByLocation(String location) throws MkRelaxException{
	ModuleModel [] mods = oldModule.getIncludeModules();
	for (int i=0;i<mods.length;i++) {
	    if (mods[i].getModuleFilename().equals(location))
		return mods[i];
	}
	throwException("Module not found : "+location);
	return null;
    }

    void setCurrentPath(String in) {
	MainApp.getAppContext().getMonitor().debug("SetCurrentPath:"+in+" (<-"+getCurrentPath()+")");
	if (StringUtil.isNull(in)) {
	    curPath = System.getProperty("user.dir");
	} else {
	    curPath = in;
	}
    }

    String getCurrentPath() {
	if (StringUtil.isNull(curPath)) {
	    return System.getProperty("user.dir");
	} else {
	    return curPath;
	}
    }

    String getCurrentParentPath() {
	if (StringUtil.isNull(curPath)) {
	    return System.getProperty("user.dir");
	} else {
	    return new File(curPath).getParent();
	}
    }

    String transCanonicalPath(String name) throws MkRelaxException {
	File f = new File(name);
	if (f.exists()) return name;
	f = new File(getCurrentParentPath(),name);
	if (f.exists()) return f.getPath();
	throwException(format("ModuleFile{0}NotFound",name));
	return null;
    }

     void prepareTables(Element moduleNode) throws MkRelaxException {
	Element [] elementRuleNodes = getElementsByTagName(moduleNode,"elementRule");
	for (int i=0;i<elementRuleNodes.length;i++) {
	    String label = getLabelOfElement(elementRuleNodes[i]);
	    ObjectModel object = ElementUtil.getObject(moduleModel,label);
	    ruleTable.put(getLabelOfElement(elementRuleNodes[i]),object);
	    tagTable.put(getRoleOfElement(elementRuleNodes[i]),object);
	}
	Element [] tagNodes = getElementsByTagName(moduleNode,"tag");
	for (int i=0;i<tagNodes.length;i++) {
	    String role = getRoleOfTag(tagNodes[i]);
	    tagNodeTable.put(role,tagNodes[i]);
	    getObjectByRole(role).setTagName(getNameOfTag(tagNodes[i]));
	}
	Element [] attPoolNodes = getElementsByTagName(moduleNode,"attPool");
	for (int i=0;i<attPoolNodes.length;i++) {
	    String role=getRoleOfAttPool(attPoolNodes[i]);
	    attPoolTable.put(role,ElementUtil.getAttributePool(role));
	}
	Element [] hedgeRuleNodes = getElementsByTagName(moduleNode,"hedgeRule");
	for (int i=0;i<hedgeRuleNodes.length;i++) {
	    String label = getLabelOfHedgeRule(hedgeRuleNodes[i]);
	    hedgeRuleTable.put(label,ElementUtil.getHedgeRule(moduleModel,label));
	}
    }

    void setupModule(Element moduleNode) throws MkRelaxException {
	moduleModel = ModuleModel.createNewModule();
	moduleModel.setModuleVersion(moduleNode.getAttribute("moduleVersion"));
	moduleModel.setRelaxCoreVersion(moduleNode.getAttribute("relaxCoreVersion"));
	moduleModel.setTargetNameSpace(moduleNode.getAttribute("targetNamespace"));
	moduleModel.setModuleVersion(moduleNode.getAttribute("moduleVersion"));
	moduleModel.setModuleFilename(getCurrentPath());
	moduleModel.setModuleComment(getDocumentation(moduleNode));
    }

    void setupInterface(Element moduleNode) throws MkRelaxException {
	pathStack.push("interface");
	Element [] temp = getElementsByTagName(moduleNode,"interface");
	if (temp != null && temp.length > 0) {
	    Element [] exports = getElementsByTagName(temp[0],"export");
	    for (int i=0;i<exports.length;i++) {
		String label = exports[i].getAttribute("label");
		ObjectModel obj = getObjectByLabel(label);
		obj.setExportable(true);
	    }
	}
	pathStack.pop();
    }

    String getAppinfo(String id,Element element) {
	String line = getAnnotationTag(element,"appinfo");
	if (!StringUtil.isNull(line) && 
	    line.indexOf(id) != -1) {
	    return line.substring(id.length());
	}
	return null;
    }

    String getDocumentation(Element element) {
	return getAnnotationTag(element,"documentation");
    }

    private String getAnnotationTag(Element element,String tagname) {
	Element [] noteNodes = getElementsByTagName(element,"annotation");
	if (noteNodes != null && noteNodes.length!=0) {
	    Element [] list = getElementsByTagName(noteNodes[0],tagname);
	    if (list != null && list.length > 0) {
		if (list[0].getFirstChild() != null) {
		    return list[0].getFirstChild().getNodeValue();
		}
	    }
	}
	return null;
    }

    void findRoots() {
	String [] refs = new String [refList.size()];
	refs = (String[])refList.toArray(refs);
	for (int i=0;i<refs.length;i++) {
	    ruleTable.remove(refs[i]);
	}
	Iterator it = ruleTable.values().iterator();
	while(it.hasNext()) {
	    ObjectModel obj = (ObjectModel)it.next();
	    moduleModel.addRootObject(obj);
	}
    }

    void setupElementRules(Element moduleNode) throws MkRelaxException {
	pathStack.push("elementRules");
	Element [] elementRuleNodes = getElementsByTagName(moduleNode,"elementRule");
	for (int i=0;i<elementRuleNodes.length;i++) {
	    String label = getLabelOfElement(elementRuleNodes[i]);
	    ObjectModel object = getObjectByLabel(label);
	    object.setComment(getDocumentation(elementRuleNodes[i]));
	    String type = elementRuleNodes[i].getAttribute("type");
	    if (!StringUtil.isNull(type)) {
		object.setType(type);
	    } else {
		setupElements(elementRuleNodes[i],object);
	    }
	    setupAttributes(elementRuleNodes[i],object);
	}
	pathStack.pop();
    }

    void setupHedgeRules(Element moduleNode) throws MkRelaxException {
	pathStack.push("hedgeRules");
	Element [] hedgeRuleNodes = getElementsByTagName(moduleNode,"hedgeRule");
	for (int i=0;i<hedgeRuleNodes.length;i++) {
	    String label = getLabelOfHedgeRule(hedgeRuleNodes[i]);
	    HedgeRuleElementModel object = getHedgeRuleByLabel(label);
	    object.setComment(getDocumentation(hedgeRuleNodes[i]));
	    moduleModel.addHedgeRule(object);
	    setupHedgeElement(hedgeRuleNodes[i],object);
	}
	pathStack.pop();
    }

    void setupElements(Element elementRule,ElementOperation parent) throws MkRelaxException {
	Element [] elements = getElementsByTagName(elementRule,"mixed");
	if (elements != null && elements.length > 0) {
	    pathStack.push("mixed");
 	    MixedModel mixed = ElementUtil.getMixed(moduleModel);
	    parent.addOpElements(ElementUtil.toArray(mixed));
	    setupHedgeElement(elements[0],mixed);
	    pathStack.pop();
	} else {
	    setupHedgeElement(elementRule,parent);
	}
    }
    
    void setupHedgeElement(Element elementRule,ElementOperation parent) throws MkRelaxException {
	Element [] elements = getChildElements(elementRule);
	if (elements == null && elements.length == 0) return;
	for (int i=0;i<elements.length;i++) {
	    if (elements[i].getTagName().equals("ref")) {
		pathStack.push("ref["+i+"]");
		String label = elements[i].getAttribute("label");
		ObjectModel object = getObjectByLabel(label);
		RefElementModel ref = ElementUtil.getRef(moduleModel,object);
		refList.add(label);
		setupOccurs(elements[i],ref);
		parent.addOpElements(ElementUtil.toArray(ref));
		pathStack.pop();
	    } else if (elements[i].getTagName().equals("hedgeRef")) {
		pathStack.push("hedgeRef["+i+"]");
		String label = elements[i].getAttribute("label");
		HedgeRuleElementModel hedgeRule = getHedgeRuleByLabel(label);
		HedgeRefElementModel ref = ElementUtil.getHedgeRef(moduleModel,hedgeRule);
		setupOccurs(elements[i],ref);
		parent.addOpElements(ElementUtil.toArray(ref));
		pathStack.pop();
	    } else if (elements[i].getTagName().equals("none")) {
		parent.addOpElements(ElementUtil.toArray(ElementUtil.getNone(moduleModel)));
	    } else if (elements[i].getTagName().equals("empty")) {
		parent.addOpElements(ElementUtil.toArray(ElementUtil.getEmpty(moduleModel)));
	    } else if (elements[i].getTagName().equals("choice")) {
		pathStack.push("choice["+i+"]");
		ChoiceElementModel choice = ElementUtil.getChoice(moduleModel);
		setupOccurs(elements[i],choice);
		parent.addOpElements(ElementUtil.toArray(choice));
		setupHedgeElement(elements[i],choice);
		pathStack.pop();
	    } else if (elements[i].getTagName().equals("sequence")) {
		pathStack.push("sequence["+i+"]");
		SequenceElementModel sequence = ElementUtil.getSequence(moduleModel);
		setupOccurs(elements[i],sequence);
		parent.addOpElements(ElementUtil.toArray(sequence));
		setupHedgeElement(elements[i],sequence);
		pathStack.pop();
	    }
	}
    }

    void setupOccurs(Element element,AbstractElementModel model) {
	if (!StringUtil.isNull(element.getAttribute("occurs")))
	    model.setOccurs(element.getAttribute("occurs"));
    }

    void setupAttributes(Element elementRule,
			 ObjectModel object) throws MkRelaxException {
	pathStack.push(object.getTagName()+"(tag)");
	Element tagNode = (Element)tagNodeTable.get
	    (getRoleOfElement(elementRule));
	AttributeModel [] attributes = getAttributes(tagNode);
	for (int i=0;i<attributes.length;i++) {
	    object.addAttribute(attributes[i]);
	}
	pathStack.pop();
    }

    void setupAttPools(Element moduleNode) throws MkRelaxException {
	pathStack.push("attPools");
	Element [] poolNodes = getElementsByTagName(moduleNode,"attPool");
	for (int i=0;i<poolNodes.length;i++) {
	    String role = getRoleOfAttPool(poolNodes[i]);
	    AttributePoolModel pool = getAttPoolByRole(role);
	    pool.setComment(getDocumentation(poolNodes[i]));
	    moduleModel.addAttributePool(pool);
	    AttributeModel [] attributes = getAttributes(poolNodes[i]);
	    for (int j=0;j<attributes.length;j++) {
		pool.addAttribute(attributes[j]);
	    }
	}
	pathStack.pop();
    }

    AttributeModel [] getAttributes(Element parentNode) throws MkRelaxException {
	List list = new ArrayList();
	Element [] attributeNodes = getElementsByTagName(parentNode,"attribute");
	for (int i=0;i<attributeNodes.length;i++) {
	    String name = attributeNodes[i].getAttribute("name");
	    String type = attributeNodes[i].getAttribute("type");
	    String req = attributeNodes[i].getAttribute("required");
	    if (StringUtil.isNull(name) )
		throwException(format("NameNotFound{0}",attributeNodes[i]));
	    if (StringUtil.isNull(type))
		throwException(format("TypeNotFound{0}",attributeNodes[i]));
	    ConcreteAttributeModel att = ElementUtil.getAttribute(name);
	    att.setType(type);
	    if (StringUtil.isNull(req))
		att.setRequired(false);
	    att.setComment(getDocumentation(attributeNodes[i]));
	    list.add(att);
	}
	Element [] attRefNodes = getElementsByTagName(parentNode,"ref");
	for (int i=0;i<attRefNodes.length;i++) {
	    String role = attRefNodes[i].getAttribute("role");
	    AttributePoolModel atp = getAttPoolByRole(role);
	    AttributeRefModel att = ElementUtil.getAttributeRef(atp);
	    list.add(att);
	}
	AttributeModel [] ret = new AttributeModel[list.size()];
	return (AttributeModel[])list.toArray(ret);
    }


     String getLabelOfElement(Element elementRule) throws MkRelaxException {
	String label = elementRule.getAttribute("label");
	if (StringUtil.isNull(label))
	    label = elementRule.getAttribute("role");
	if (StringUtil.isNull(label))
	    throwException(format("RoleNotFound{0}",elementRule));
	return label;
    }

     String getRoleOfElement(Element elementRule) throws MkRelaxException {
	String label = elementRule.getAttribute("role");
	if (StringUtil.isNull(label))
	    throwException(format("RoleNotFound{0}",elementRule));
	return label;
    }

     String getNameOfTag(Element tag) throws MkRelaxException {
	String role = tag.getAttribute("name");
	if (StringUtil.isNull(role))
	    throwException(format("NameNotFound{0}",tag));
	return role;
    }

     String getRoleOfTag(Element tag) throws MkRelaxException {
	String role = tag.getAttribute("role");
	if (StringUtil.isNull(role))
	    role = tag.getAttribute("name");
	if (StringUtil.isNull(role))
	    throwException(format("NameNotFound{0}",tag));
	return role;
    }

     String getRoleOfAttPool(Element attPool) throws MkRelaxException {
	String role = attPool.getAttribute("role");
	if (StringUtil.isNull(role))
	    throwException(format("RoleNotFound{0}",attPool));
	return role;
    }

     String getLabelOfHedgeRule(Element hedgeRule) throws MkRelaxException {
	String label = hedgeRule.getAttribute("label");
	if (StringUtil.isNull(label))
	    throwException(format("LabelNot{0}Found",hedgeRule));
	return label;
    }

    ObjectModel getObjectByRole(String role) throws MkRelaxException {
	ObjectModel obj = (ObjectModel)tagTable.get(role);
	if (obj == null)
	    throwException(format("ElementTag{0}NotFound",role));
	return obj;
    }

    ObjectModel getObjectByLabel(String label) throws MkRelaxException {
	ObjectModel obj = (ObjectModel)ruleTable.get(label);
	if (obj == null) obj = getObjectByLabelFromIncluded(label);
	if (obj == null)
	    throwException(format("ElementRule{0}NotFound",label));
	return obj;
    }

    ObjectModel getObjectByLabelFromIncluded(String label) {
	ModuleModel [] incs = moduleModel.getIncludeModules();
	for (int i=0;i<incs.length;i++) {
	    ObjectModel obj = incs[i].getObjectByLabel(label);
	    if (obj != null) return obj;
	}
	return null;
    }

    HedgeRuleElementModel getHedgeRuleByLabel(String label) throws MkRelaxException {
	HedgeRuleElementModel obj = (HedgeRuleElementModel)hedgeRuleTable.get(label);
	if (obj == null) obj = getHedgeRuleByLabelFromIncluded(label);
	if (obj == null)
	    throwException(format("HedgeRule{0}NotFound",label));
	return obj;
    }

    HedgeRuleElementModel getHedgeRuleByLabelFromIncluded(String label) {
	ModuleModel [] incs = moduleModel.getIncludeModules();
	for (int i=0;i<incs.length;i++) {
	    HedgeRuleElementModel obj = incs[i].getHedgeRuleByLabel(label);
	    if (obj != null) return obj;
	}
	return null;
    }


    AttributePoolModel getAttPoolByRole(String role) throws MkRelaxException {
	AttributePoolModel obj = (AttributePoolModel)attPoolTable.get(role);
	if (obj == null) obj = getAttPoolByRoleFromIncluded(role);
	if (obj == null)
	    throwException(format("AttPool{0}NotFound",role));
	return obj;
    }
    
    AttributePoolModel getAttPoolByRoleFromIncluded(String role) {
	ModuleModel [] incs = moduleModel.getIncludeModules();
	for (int i=0;i<incs.length;i++) {
	    AttributePoolModel obj = incs[i].getAttributePoolByRole(role);
	    if (obj != null) return obj;
	}
	return null;
    }


    //
    // Expanding RELAX
    //
    //

     void expandRelax(Document document) throws MkRelaxException {
	Element module = document.getDocumentElement();
	Element [] nodes = getElementsByTagName(module,"elementRule");
	for (int i=0;i<nodes.length;i++) {
	    expandElementGen(document,module,nodes[i]);
	}
	nodes = getElementsByTagName(module,"hedgeRule");
	for (int i=0;i<nodes.length;i++) {
	    expandElementGen(document,module,nodes[i]);
	}
	nodes = getElementsByTagName(module,"elementRule");
	for (int i=0;i<nodes.length;i++) {
	    Element [] tags = getElementsByTagName(nodes[i],"tag");
	    if (tags.length>0) {
		expandTag(document,module,nodes[i],tags[0]);
	    }
	}
    }
    
     void expandTag(Document document,Element module,
		    Element elementRule,Element tag) throws MkRelaxException{
	Element ntag = document.createElement("tag");
	if (StringUtil.isNull(tag.getAttribute("name"))) 
	    ntag.setAttribute("name",getLabelOfElement(elementRule));
	else
	    ntag.setAttribute("name",tag.getAttribute("name"));
	String trole = getUniqueRole
	    (module,getLabelOfElement(elementRule));
	ntag.setAttribute("role",trole);
	elementRule.setAttribute("role",trole);
	Element [] atList = getChildElements(tag);
	for (int i=0;i<atList.length;i++) {
	    Element att = null;
	    if (atList[i].getTagName().equals("attribute")) {
		att = document.createElement("attribute");
		att.setAttribute("name",atList[i].getAttribute("name"));
		att.setAttribute("type",atList[i].getAttribute("type"));
		if (!StringUtil.isNull(atList[i].getAttribute("required"))) {
		    att.setAttribute("required",
				     atList[i].getAttribute("required"));
		}
		appendAnnotation(document,att,getDocumentation(atList[i]));
	    } else if (atList[i].getTagName().equals("ref")) {
		att = document.createElement("ref");
		att.setAttribute("role",atList[i].getAttribute("role"));
	    } else throwException(format("WrongTag{0}",atList[i]));
	    ntag.appendChild(att);
	}
	elementRule.removeChild(tag);
	module.appendChild(ntag);
    }

    void expandElementGen(Document document,Element module,
			  Element elementRule) {
	expandElement(document,module,elementRule);
	Element [] children = getChildElements(elementRule);
	for (int i=0;i<children.length;i++) {
	    if (children[i].getNodeName().equals("sequence") ||
		children[i].getNodeName().equals("choice") ||
		children[i].getNodeName().equals("mixed")) {
		expandElementGen(document,module,children[i]);
	    }
	}
    }

     void expandElement(Document document,Element module,
			Element elementRule) {
	Element [] elements = getElementsByTagName(elementRule,"element");
	if (elements.length > 0) {
	    for (int i=0;i<elements.length;i++) {
		Element ref = document.createElement("ref");
		String tlabel = getUniqueLabel(module,elements[i].getAttribute("name"));
		ref.setAttribute("label",tlabel);
		if (!StringUtil.isNull(elements[i].getAttribute("occurs")))
		    ref.setAttribute("occurs",elements[i].getAttribute("occurs"));
		elementRule.appendChild(ref);
		elementRule.removeChild(elements[i]);
		Element newElementRule = document.createElement("elementRule");
		appendAnnotation(document,newElementRule,getDocumentation(elements[i]));
		newElementRule.setAttribute("label",tlabel);
		newElementRule.setAttribute("role",tlabel);
		newElementRule.setAttribute
		    ("type",elements[i].getAttribute("type"));
		module.appendChild(newElementRule);
		Element ntag = document.createElement("tag");
		ntag.setAttribute("name",elements[i].getAttribute("name"));
		ntag.setAttribute("role",tlabel);
		module.appendChild(ntag);
	    }
	} else {
	    Element [] list = getChildElements(elementRule);
	    for (int i=0;i<list.length;i++) {
		expandElement(document,module,list[i]);
	    }
	}
    }

    protected Element appendAnnotation(Document document,Element element,
				       String comment) {
	Element an = document.createElement("annotation");
	Element dd = document.createElement("documentation");
	dd.appendChild( document.createTextNode( comment ));
	an.appendChild( dd );
	element.appendChild( an );
	return an;
    }

     Element [] getElementsByTagName(Element node,String name) {
	Element [] es = getChildElements(node);
	List list = new ArrayList();
	for (int i=0;i<es.length;i++) {
	    if (es[i].getNodeName().equals(name))
		list.add(es[i]);
	}
	Element [] els = new Element[list.size()];
	return (Element[])list.toArray(els);
    }

     Element [] getChildElements(Node node) {
	NodeList list = node.getChildNodes();
	List rets = new ArrayList();
	for (int i=0;i<list.getLength();i++) {
	    if (list.item(i) instanceof Element) 
		rets.add(list.item(i));
	}
	Element [] els = new Element[rets.size()];
	return (Element[])rets.toArray(els);
    }

    Document readDocumentFromFile(String filename) throws MkRelaxException {
	try {
	    return getDocumentBuilder().parse(new File(filename));
	} catch (RuntimeException e) {
	    throwException(MainApp.getResString("RelaxSyntaxError")+" : "+e.getMessage());
	} catch (SAXException e) {
	    throwException(MainApp.getResString("XMLException")+" : "+e.getMessage());
	} catch (IOException e) {
	    throwException(MainApp.getResString("RelaxIOException")+" : "+e.getMessage());
	}
	return null;
    }

    Document readDocumentFromString(String input) throws MkRelaxException {
	try {
	    return getDocumentBuilder().parse(new InputSource(new StringReader(input)));
	} catch (RuntimeException e) {
	    throwException(MainApp.getResString("RelaxSyntaxError")+" : "+e.getMessage());
	} catch (SAXException e) {
	    throwException(MainApp.getResString("XMLException")+" : "+e.getMessage());
	} catch (IOException e) {
	    throwException(MainApp.getResString("RelaxIOException")+" : "+e.getMessage());
	}
	return null;
    }

    DocumentBuilder getDocumentBuilder() throws MkRelaxException {
	try {
	    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
	    DocumentBuilder db = dbf.newDocumentBuilder();
	    db.setEntityResolver(new RELAXEntityResolver());
	    return db;
	} catch (ParserConfigurationException e) {
	    throwException(MainApp.getResString("ParseConfigurationException")+" : "+e.getMessage());
	}
	return null;
    }

     String getUniqueLabel(Element module,String input) {
	String ret = input;
	long no = 0;
	while(true) {
	    if (isUnique(module,"element","label",ret) &&
		isUnique(module,"elementRule","label",ret) &&
		isUnique(module,"hedgeRule","label",ret)) break;
	    ret = input+"_"+no;
	}
	return ret;
    }

     String getUniqueRole(Element module,String input) {
	String ret = input;
	long no = 0;
	while(true) {
	    if (isUnique(module,"attPool","role",ret) &&
		isUnique(module,"tag","role",ret)) break;
	    ret = input+"_"+no;
	    if (no > 20) {
		MainApp.getAppContext().getMonitor().warning("too many non-unique roles. : "+input);
	    }
	}
	return ret;
    }

     boolean isUnique(Element module,String tagName,
				 String att,String input) {
	Element [] entry = getAllElement(module,tagName);
	for (int i=0;i<entry.length;i++) {
	    String sample = entry[i].getAttribute(att);
	    if (sample.equals(input)) return false;
	}
	return true;
    }

     Element [] getAllElement(Element module,String tagName) {
	NodeList list = module.getElementsByTagName(tagName);
	List entry = new ArrayList();
	for (int i=0;i<list.getLength();i++) {
	    if (list.item(i) instanceof Element) {
		entry.add(list.item(i));
	    }
	}
	Element [] ret = new Element[entry.size()];
	return (Element[])entry.toArray(ret);
    }

    public static String getDumpModule(ModuleModel m) {
	StringBuffer sb = new StringBuffer("\n");
	sb.append("module:"+m.getModuleName()).append("\n");
	sb.append("filename:"+m.getModuleFilename()).append("\n");
	ObjectModel [] objects = m.getExportableObjects();
	for (int j=0;j<objects.length;j++) {
	    sb.append("export:"+objects[j].getLabel()).append("\n");
	}
	objects = m.getObjects();
	for (int j=0;j<objects.length;j++) {
	    sb.append("element:"+objects[j].getLabel()).append("\n");
	}
	HedgeRuleElementModel [] hedgeRules = m.getHedgeRules();
	for (int j=0;j<hedgeRules.length;j++) {
	    sb.append("hedgeRule:"+hedgeRules[j].getLabel()).append("\n");
	}
	AttributePoolModel [] attPools = m.getAttributePools();
	for (int j=0;j<attPools.length;j++) {
	    sb.append("attPool:"+attPools[j].getRole()).append("\n");
	}
	objects = m.getAllObjects();
	for (int j=0;j<objects.length;j++) {
	    sb.append("allElement:"+objects[j].getLabel()).append("\n");
	}
	hedgeRules = m.getAllHedgeRules();
	for (int j=0;j<hedgeRules.length;j++) {
	    sb.append("allHedgeRule:"+hedgeRules[j].getLabel()).append("\n");
	}
	attPools = m.getAllAttributePools();
	for (int j=0;j<attPools.length;j++) {
	    sb.append("allAttPool:"+attPools[j].getRole()).append("\n");
	}
	return sb.toString();
    }

    String format(String mesRes,Element arg) {
	return format(mesRes,arg.toString());
    }
    String format(String mesRes,String arg) {
	return MessageFormat.format(MainApp.getResString(mesRes),
				    new String [] {arg});
    }

    void throwException(String message) throws MkRelaxException {
	String path = "";
	while(pathStack.size()>0) {
	    path = pathStack.pop()+">"+path;
	}
	String output = "\n [error path : "+path+"]";
	throw new MkRelaxException(message+output);
    }

    public static String document2string(Document document) {
	StringWriter out = new StringWriter();
	OutputFormat format = new OutputFormat(document);
	format.setIndenting(true);
	XMLSerializer ser = new XMLSerializer(out,format);
	try {
	    ser.serialize(document);
	} catch (IOException e) {
	    throw new RuntimeException("IOException? : "+e.getMessage());
	}
	return out.toString();
    }

    public static void main(String [] args) {
	try {
	    ModuleModel m = RelaxReader.getModuleFromFile(args[0]);
	    System.out.println(m.getRelax());
	    System.out.println(getDumpModule(m));
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }
}
