/* MkRelax, Visual Relax Editor
 *  Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax;

import java.util.Stack;
import java.util.ArrayList;


public class VisitorCenter {

    ModuleModel module;
    Visitor visitor;
    int loop = 0;
    Stack objectStack = new Stack();
    boolean traverseObject = true;
    boolean traverseHedgeRule = false;
    boolean insideOnly = false;
    ArrayList encounteredObjects = new ArrayList();

    public static Visitor traverseObjectAroundAllModules(ModuleModel m,Visitor v) {
	traverseObject(m,v);
	ModuleModel [] ms = m.getIncludeModules();
	for (int i=0;i<ms.length;i++) {
	    traverseObject(ms[i],v);
	}
	return v;
    }

    public static Visitor traverseHedgeRuleAroundAllModules(ModuleModel m,Visitor v) {
	traverseHedgeRule(m,v);
	ModuleModel [] ms = m.getIncludeModules();
	for (int i=0;i<ms.length;i++) {
	    traverseHedgeRule(ms[i],v);
	}
	return v;
    }

    public static Visitor traverseAllAroundModules(ModuleModel m,Visitor v) {
	traverseAll(m,v);
	ModuleModel [] ms = m.getIncludeModules();
	for (int i=0;i<ms.length;i++) {
	    traverseAll(ms[i],v);
	}
	return v;
    }

    // all objects in current module and 
    // referenced objects in other modules.
    public static Visitor traverseObject(ModuleModel m,Visitor v) {
	VisitorCenter vc = new VisitorCenter(m,v);
	vc.startVisitor();
	return v;
    }

    // all hedges in current module and 
    // referenced objects in other modules.
    public static Visitor traverseHedgeRule(ModuleModel m,Visitor v) {
	VisitorCenter vc = new VisitorCenter(m,v,false,true,false);
	vc.startVisitor();
	return v;
    }

    // all objects and hedges in current module and 
    // referenced objects in other modules.
    public static Visitor traverseAll(ModuleModel m,Visitor v) {
	VisitorCenter vc = new VisitorCenter(m,v,true,true,false);
	vc.startVisitor();
	return v;
    }

    // all objects and hedges in only current module
    public static Visitor traverseAllInsideObjects(ModuleModel m,Visitor v) {
	VisitorCenter vc = new VisitorCenter(m,v,true,true,true);
	vc.startVisitor();
	return v;
    }

    protected VisitorCenter(ModuleModel module,Visitor visitor,
			    boolean traverseObject,
			    boolean traverseHedgeRule,
			    boolean insideOnly) {
	this.loop = visitor.getAllowedNestedLoop();
	this.module = module;
	this.visitor = visitor;
	this.traverseObject = traverseObject;
	this.traverseHedgeRule = traverseHedgeRule;
	this.insideOnly = insideOnly;
    }

    protected VisitorCenter(ModuleModel module,Visitor visitor) {
	this(module,visitor,true,false,false);
    }

    protected void startVisitor() {
	visitor.start(module);
	if (traverseObject) {
	    ObjectModel [] roots = module.getRootObjects();
	    for (int i=0;i<roots.length;i++) {
		MainApp.getAppContext().getMonitor().debugStackTrace
		    ("VISITOR:root>"+roots[i].getLabel());
		root(roots[i]);
	    }
	}
	if (traverseHedgeRule) {
	    HedgeRuleElementModel [] roots = module.getHedgeRules();
	    for (int i=0;i<roots.length;i++) {
		hedgeRuleRoot(roots[i]);
	    }
	}
	visitor.end(module);
    }

    protected void hedgeRuleRoot(HedgeRuleElementModel root) {
	hedgeRule(root);
    }

    protected void hedgeRule(HedgeRuleElementModel object) {
	if (object == null) return;
	if (insideOnly && object.getParentModule() != module) return;
	//stack push
	visitor.hedgeRuleEnter(object);
	objectStack.push(object);
	// branch
	AbstractHedgeModel element = object.getElement();
	if (element != null) {
	    if (element instanceof AbstractHedgeModel) {
		hedge((AbstractHedgeModel)element);
	    } else {
		throw new InternalError("No such element ["+element+"]");
	    }
	}
	objectStack.pop();
	visitor.hedgeRuleExit(object);
    }

    protected void root(ObjectModel object) {
	if (object == null) throw new InternalError("Root object is null!");
	visitor.root(object);
	object(object);
    }

    protected void object(ObjectModel object) {
	if (object == null) return;
	if (insideOnly && object.getParentModule() != module) return;
	//first contact?
	if (encounteredObjects.contains(object)) {
	    if (visitor.isOnlyFirstContact())
		return;
	} else {
	    encounteredObjects.add(object);
	}
	//forbid loop reference
	if (objectStack.contains(object) &&
	    visitor.getAllowedNestedLoop() <= 0) {
	    return;
	}
	//nested loop check
	int num = countObjectNum(object);
	if (visitor.getAllowedNestedLoop() > 0 &&
	    num >= visitor.getAllowedNestedLoop()) {
	    return;
	}
	//stack push
	objectStack.push(object);
	visitor.objectEnter(object,num);
	// branch
	ElementModel element = object.getElement();
	if (element != null) {
	    if (element instanceof MixedModel) {
		mixed((MixedModel)element);
	    } else if (element instanceof AbstractHedgeModel) {
		hedge((AbstractHedgeModel)element);
	    } else {
		throw new InternalError("No such element ["+element+"]");
	    }
	}
	objectStack.pop();
	visitor.objectExit(object);
    }

    int countObjectNum(ObjectModel obj) {
	if (objectStack.contains(obj)) {
	    int num = 0;
	    for (int i=0;i<objectStack.size();i++) {
		if (objectStack.get(i) == obj)
		    num++;
	    }
	    return num;
	}else {
	    return 0;
	}
    }

    protected void hedge(AbstractHedgeModel element) {
	if (element == null) return;
	if (element instanceof NoneElementModel) {
	    none((NoneElementModel)element);
	} else if (element instanceof EmptyElementModel) {
	    empty((EmptyElementModel)element);
	} else if (element instanceof ChoiceElementModel) {
	    choice((ChoiceElementModel)element);
	} else if (element instanceof SequenceElementModel) {
	    sequence((SequenceElementModel)element);
	} else if (element instanceof HedgeRefElementModel) {
	    hedgeRef((HedgeRefElementModel)element);
	} else if (element instanceof RefElementModel) {
	    ref((RefElementModel)element);
	} else {
	    throw new InternalError("No such element ["+element+"]");
	}
    }

    protected void mixed(MixedModel mixed) {
	visitor.mixedEnter(mixed);
	hedge(mixed.getElement());
	visitor.mixedExit(mixed);
    }

    protected void none(NoneElementModel none) {
	visitor.none(none);
    }
    
    protected void empty(EmptyElementModel empty) {
	visitor.empty(empty);
    }

    protected void choice(ChoiceElementModel element) {
	visitor.choiceEnter(element);
	AbstractHedgeModel [] hedges = element.getElements();
	for (int i=0;i<hedges.length;i++) {
	    hedge(hedges[i]);
	}
	visitor.choiceExit(element);
    }

    protected void sequence(SequenceElementModel element) {
	visitor.sequenceEnter(element);
	AbstractHedgeModel [] hedges = element.getElements();
	for (int i=0;i<hedges.length;i++) {
	    hedge(hedges[i]);
	}
	visitor.sequenceExit(element);
    }

    protected void hedgeRef(HedgeRefElementModel element) {
	visitor.hedgeRef(element);
    }

    protected void ref(RefElementModel element) {
	visitor.ref(element);
	object(element.getAt());
    }
}
