/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax.gui;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import javax.swing.tree.TreeNode;
import javax.swing.tree.DefaultTreeModel;
import jp.gr.java_conf.ccs2.comp.MessageListener;
import jp.gr.java_conf.ccs2.tool.mkrelax.*;


public abstract class AbstractObjectTreeModel extends DefaultTreeModel {

	ModuleModel mainModule;
	TNElement mainRoot;
	MessageListener messageListener;

	public AbstractObjectTreeModel(ModuleModel mm,MessageListener ms) {
		super(new TNModule(mm,ms));
		mainModule = mm;
		messageListener = ms;
		setupRoots();
	}

	protected abstract TNElement createRootNode(ModuleModel m);
	protected abstract TNElement createIncludedNode(ModuleModel m,
													TNElement parent);

	private void setupRoots() {
		TNElement mainRoot = createRootNode(mainModule);
		setupRootNode(mainModule,mainRoot);
		setRoot(mainRoot);
		ModuleModel [] incs = mainModule.getIncludeModules();
		for (int i=0;i<incs.length;i++) {
			TNElement rt = createIncludedNode(incs[i],mainRoot);
			setupRootNode(incs[i],rt);
			mainRoot.addChild(rt);
		}
	}

	protected abstract void traverse(ModuleModel m,Visitor v);
	protected abstract void traverseAll(ModuleModel m,Visitor v);

	private void setupRootNode(ModuleModel curModule,TNElement root) {
		AVisitor av = new AVisitor(root);
		if (curModule == mainModule)
			traverse(curModule,av);
		else 
			traverseAll(curModule,av);
	}

	public TNElement getMainRootNode() {
		return mainRoot;
	}

	public MessageListener getMessageListener() {
		return messageListener;
	}

	class AVisitor extends VisitorClass {

		TNElement curRoot;

		AVisitor(TNElement cr) {
			curRoot= cr;
		}

		protected TNElement getCurrentRoot() {
			return curRoot;
		}

		Stack nodeStack = new Stack();
		RefElementModel ref;
		List currentChildren = new ArrayList();
		Stack childrenStack = new Stack();

		public int getAllowedNestedLoop() {
			return 1;
		}

		public void start(ModuleModel m){
			nodeStack.push(getCurrentRoot());
		}

		public void end(ModuleModel m){
			nodeStack.pop();
			getCurrentRoot().setElements(children());
		}
	
		public void root(ObjectModel obj){
		}

		void pushChildren() {
			childrenStack.push(currentChildren);
			currentChildren = new ArrayList();
		}

		void popChildren() {
			currentChildren = (List)childrenStack.pop();
		}

		TNElement curParent() {
			return (TNElement)nodeStack.peek();
		}

		TreeNode [] children() {
			if (currentChildren == null) return new TreeNode[0];
			TreeNode [] ret = new TreeNode[currentChildren.size()];
			return (TreeNode[])currentChildren.toArray(ret);
		}

		void begin(TNElement tn) {
			nodeStack.push(tn);
			pushChildren();
		}

		void end() {
			TNElement node = (TNElement)nodeStack.pop();
			node.setElements(children());
			popChildren();
			currentChildren.add(node);
		}

		public void hedgeRuleEnter(HedgeRuleElementModel obj){
			begin(new TNHedgeRule(curParent(),messageListener,obj));
		}

		public void hedgeRuleExit(HedgeRuleElementModel obj){
			end();
		}

		public void objectEnter(ObjectModel obj,int loop){
			TNObject node = null;
			if (loop==getAllowedNestedLoop()) {//nested shadow object
				node = new TNObject(curParent(),getMessageListener(),obj,true);
			} else {
				node = new TNObject(curParent(),getMessageListener(),obj);
			}
			if (ref != null) node.setRef(ref);
			ref = null;
			begin(node);
		}
		public void objectExit(ObjectModel obj){
			TNObject node = (TNObject)nodeStack.pop();
			if (!node.isShadow()) {
				node.setElements(children());
			}
			popChildren();
			currentChildren.add(node);
		}

		public void mixedEnter(MixedModel obj){
			begin( new TNMixed(curParent(),getMessageListener(),obj));
		}
		public void mixedExit(MixedModel obj){
			end();
		}

		public void none(NoneElementModel obj){
			TNElement node = new TNNone(curParent(),getMessageListener(),obj);
			currentChildren.add(node);
		}
		public void empty(EmptyElementModel obj){
			TNElement node = new TNEmpty(curParent(),getMessageListener(),obj);
			currentChildren.add(node);
		}

		public void choiceEnter(ChoiceElementModel obj){
			begin( new TNChoice(curParent(),getMessageListener(),obj));
		}
		public void choiceExit(ChoiceElementModel obj){
			end();
		}

		public void sequenceEnter(SequenceElementModel obj){
			begin( new TNSequence(curParent(),getMessageListener(),obj));
		}
		public void sequenceExit(SequenceElementModel obj){
			end();
		}
	
		public void hedgeRef(HedgeRefElementModel obj){
			TNElement node = new TNHedgeRef(curParent(),getMessageListener(),obj);
			currentChildren.add(node);
		}
		public void ref(RefElementModel obj){
			ref = obj;
		}
	}
}
