/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.text.MessageFormat;
import javax.swing.*;
import javax.swing.KeyStroke;
import javax.swing.border.EtchedBorder;
import jp.gr.java_conf.ccs2.comp.MessageListener;
import jp.gr.java_conf.ccs2.comp.Shader;
import jp.gr.java_conf.ccs2.comp.SwingUtil;
import jp.gr.java_conf.ccs2.util.ExtensionFileFilter;
import jp.gr.java_conf.ccs2.util.StringUtil;
import jp.gr.java_conf.ccs2.tool.mkrelax.*;
import jp.gr.java_conf.ccs2.tool.mkrelax.relaxer.RelaxerDialog;

public class MainFrame extends JFrame {

	ModulePanel modulePanel;
	RelaxerDialog relaxerDialog;

	public MainFrame(ModulePanel mp) {
		super(getRes("frameTitle"));
		setModulePanel(mp);
		initGui();
	}

	private static String getRes(String res) {
		return MainApp.getResString(res);
	}

	public void setModulePanel(ModulePanel mp ){
		modulePanel = mp;
		modulePanel.setCompListener(compListener);
	}

	public void initGui() {
		Container root = getContentPane();
		root.setLayout(new BorderLayout());
		/*
		root.add(new JLabel(getRes("copyright")),
				 BorderLayout.SOUTH);
		*/
		root.add(modulePanel,BorderLayout.CENTER);

		initMenu();
		initToolbar();

		addWindowListener (new WindowAdapter () {
				public void windowClosing (WindowEvent evt) {
					exitModule();
				}
			});
	}

	JMenuItem copyMenuItem,cutMenuItem,pasteMenuItem,undoMenuItem;
	JMenuItem removeThisMenuItem,removeBranchMenuItem,relaxerMenuItem;
	void initMenu() {
		JMenuBar menubar = new JMenuBar();
		setJMenuBar(menubar);

		JMenu menu;
		//File menu
		menu = new JMenu(getRes("File")+"(F)");
		menu.setMnemonic(KeyEvent.VK_F);
		JMenuItem item = GuiUtil.getMenuItem
			("NewModule","NewModule",newAction,
			 getRes("IconFileNew"),Boolean.TRUE);
		item.setAccelerator(KeyStroke.getKeyStroke
							(KeyEvent.VK_N, ActionEvent.CTRL_MASK));
		menu.add(item);
		menu.addSeparator();
		item =GuiUtil.getMenuItem
			("OpenModule","OpenModule",loadAction,
			 getRes("IconFileOpen"),Boolean.TRUE);
		item.setAccelerator(KeyStroke.getKeyStroke
							(KeyEvent.VK_O, ActionEvent.CTRL_MASK));
		menu.add(item);
		item = GuiUtil.getMenuItem
			("SaveModule","SaveModule",saveAction,
			 getRes("IconFileSave"),Boolean.TRUE);
		item.setAccelerator(KeyStroke.getKeyStroke
							(KeyEvent.VK_S, ActionEvent.CTRL_MASK));
		menu.add(item);
		menu.add(GuiUtil.getMenuItem
				 ("SaveAsModule","SaveAsModule",saveAsAction,
				  getRes("IconFileSaveAs"),Boolean.TRUE));
		menu.addSeparator();
		item = GuiUtil.getMenuItem
			("ExecRelaxer","ExecRelaxer",execRelaxerAction,
			 getRes("IconExecRelaxer"),Boolean.TRUE);
		item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0 ));
		menu.add(item);
		menu.addSeparator();
		menu.add(GuiUtil.getMenuItem
				 ("ExitModule","ExitModule",exitAction,
				  getRes("IconFileExit"),Boolean.TRUE));
		menubar.add(menu);

		//Edit menu
		menu = new JMenu(getRes("Edit")+"(E)");
		menu.setMnemonic(KeyEvent.VK_E);
		menu.add( undoMenuItem = GuiUtil.getMenuItem
				  ("Undo","Undo",undoAction,
				   getRes("IconEditUndo"),Boolean.FALSE) );
		undoMenuItem.setAccelerator(KeyStroke.getKeyStroke
									(KeyEvent.VK_Z, ActionEvent.CTRL_MASK));
		menu.addSeparator();
		menu.add(copyMenuItem = GuiUtil.getMenuItem
				 ("Copy","Copy",copyAction,
				  getRes("IconEditCopy"),Boolean.FALSE));
		copyMenuItem.setAccelerator(KeyStroke.getKeyStroke
									(KeyEvent.VK_C, ActionEvent.CTRL_MASK));
		menu.add(cutMenuItem = GuiUtil.getMenuItem
				 ("Cut","Cut",cutAction,
				  getRes("IconEditCut"),Boolean.FALSE));
		cutMenuItem.setAccelerator(KeyStroke.getKeyStroke
								   (KeyEvent.VK_X, ActionEvent.CTRL_MASK));
		menu.add(pasteMenuItem = GuiUtil.getMenuItem
				 ("Paste","Paste",pasteAction,
				  getRes("IconEditPaste"),Boolean.FALSE));
		pasteMenuItem.setAccelerator(KeyStroke.getKeyStroke
									 (KeyEvent.VK_V, ActionEvent.CTRL_MASK));
		menu.addSeparator();
		menu.add(removeThisMenuItem = GuiUtil.getMenuItem
				 ("RemoveConnect","RemoveT",removeThisAction,
				  getRes("IconEditRemoveThis"),Boolean.FALSE));
		removeThisMenuItem.setAccelerator(KeyStroke.getKeyStroke
										  (KeyEvent.VK_DELETE, ActionEvent.CTRL_MASK));
		menu.add(removeBranchMenuItem = GuiUtil.getMenuItem
				 ("RemoveBranch","RemoveB",removeBranchAction,
				  getRes("IconEditRemoveBranch"),Boolean.FALSE));
		removeBranchMenuItem.setAccelerator(KeyStroke.getKeyStroke
											(KeyEvent.VK_DELETE, 0));
		menubar.add(menu);

		//HELP and another
		menu = new JMenu(getRes("Help")+"(H)");
		menu.setMnemonic(KeyEvent.VK_H);
		menu.add(GuiUtil.getMenuItem
				 ("About","About",aboutAction,
				  getRes("IconHelpAbout"),Boolean.TRUE));
		menubar.add(Box.createHorizontalGlue());
		menubar.add(menu);
	}

	void initToolbar() {
	
	}


	ActionListener newAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				newModule();
			}
		};
	ActionListener exitAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				exitModule();
			}
		};

	ActionListener saveAsAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				saveAsModule();
			}
		};

	ActionListener loadAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				loadModule();
			}
		};
	
	ActionListener saveAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				saveModule();
			}
		};

	ActionListener execRelaxerAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				execRelaxer();
			}
		};

	ActionListener aboutAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				aboutDialog();
			}
		};

	ActionListener undoAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				undoModule();
			}
		};
	ActionListener copyAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				copyElement();
			}
		};
	ActionListener cutAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				cutElement();
			}
		};
	ActionListener pasteAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				pasteElement();
			}
		};
	ActionListener removeThisAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				removeThisElement();
			}
		};
	ActionListener removeBranchAction = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				removeBranchElement();
			}
		};

	void undoModule() {
		modulePanel.undo();
	}

	void removeThisElement() {
		TNCommon node = modulePanel.getSelectedNode();
		if (node instanceof TNElement)
			((TNElement)node).removeConnect(this,false);
		else
			throw new InternalError("Forbidden opration.");
	}
	void removeBranchElement() {
		modulePanel.getSelectedNode().removeBranch(this,false);
	}

	TNCommon copiedNode;
	void copyElement() {
		copiedNode = modulePanel.getSelectedNode();
	}
	void cutElement() {
		copiedNode = modulePanel.getSelectedNode();
		copiedNode.removeBranch(this,true);
	}
	void pasteElement() {
		if (copiedNode == null) 
			throw new InternalError("Copied buffer is null.");
		if (modulePanel.getSelectedNode() == null)
			throw new InternalError("Selected node is null.");
		if (copiedNode instanceof TNElement && 
			modulePanel.getSelectedNode() instanceof TNElement) {
			pasteElementIntoElement();
		} else if (copiedNode instanceof TNAttribute && 
				   modulePanel.getSelectedNode() instanceof TNElement){
			pasteAttributeIntoElement();
		}
	}
	private void pasteElementIntoElement() {
		ElementOperation cpd = ((TNElement)copiedNode).getElementOperation();
		ElementOperation sel = ((TNElement)modulePanel.getSelectedNode()).getElementOperation();
		if (sel.canAddElement() && sel.acceptOpElement(cpd) ) {
			sel.addOpElements(ElementUtil.toArray(cpd.getOpCopy()));
			modulePanel.getSelectedNode().sendTreeChangedMessage();
		} else {
			throw new InternalError("Forbidden operation.");
		}
	}
	private void pasteAttributeIntoElement() {
		ObjectModel obj = (ObjectModel)(modulePanel.getSelectedNode().getObject());
		AttributeModel att = ((TNAttribute)copiedNode).getAttribute();
		obj.addAttribute(att.getCopy());
		modulePanel.getSelectedNode().sendTreeChangedMessage();
	}

	void execRelaxer() {
		if (relaxerDialog==null || !relaxerDialog.isShown()) {
			if (StringUtil.isNull(modulePanel.getModuleFilename())) {
				saveAsModule();
				if (StringUtil.isNull(modulePanel.getModuleFilename())) return;
			}
			relaxerDialog = new RelaxerDialog(modulePanel.getModuleFilename());
			relaxerDialog.show();
		} else {
			relaxerDialog.show();
		}
	}


	void newModule() {
		modulePanel.newModule();
	}

	void saveAsModule() {
		JFileChooser fc = new JFileChooser(System.getProperty("user.dir"));
		fc.setFileFilter(MainApp.getRelaxFileFilter());
		int returnVal = fc.showSaveDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fc.getSelectedFile();
			String filename = file.getPath();
			if ( file.getName().indexOf(".")==-1 ) {
				filename += ".rlx";
			}
			modulePanel.saveModule(filename);
		}
	}

	void saveModule() {
		if (StringUtil.isNull(modulePanel.getModuleFilename())) {
			saveAsModule();
		} else {
			modulePanel.saveModule(modulePanel.getModuleFilename());
		}
	}

	void loadModule() {
		JFileChooser fc = new JFileChooser(System.getProperty("user.dir"));
		fc.setFileFilter(MainApp.getRelaxFileFilter());
		int returnVal = fc.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File file = fc.getSelectedFile();
			modulePanel.loadModule(file.getPath());
			if (relaxerDialog != null) relaxerDialog.dispose();
		}
	}

	void exitModule() {
		if (modulePanel.isModuleModified()) {
			String [] arg = {modulePanel.getModuleName()};
			String mes = MessageFormat.format
				(getRes("OnExitDoYouSaveModule{0}?"),arg);
			int ret = JOptionPane.showConfirmDialog
				(this,mes,
				 getRes("ExitConfirm"),
				 JOptionPane.YES_NO_OPTION);
			if (ret == JOptionPane.YES_OPTION) {
				saveModule();
			}
		}
		System.exit(0);
	}

	void aboutDialog() {
		AboutDialog a = new AboutDialog(this);
		a.show();
	}

	MessageListener compListener = new MessageListener() {
		
			void disableAllMenu() {
				cutMenuItem.setEnabled(false);
				copyMenuItem.setEnabled(false);
				pasteMenuItem.setEnabled(false);
				removeThisMenuItem.setEnabled(false);
				removeBranchMenuItem.setEnabled(false);
			}
		
			void moduleSelected() {
				pasteMenuItem.setEnabled(false);
				removeThisMenuItem.setEnabled(false);
				cutMenuItem.setEnabled(false);
				copyMenuItem.setEnabled(false);
				removeBranchMenuItem.setEnabled(false);
			}

			void elementSelected(TNElement tnc) {
				ElementOperation com = tnc.getElementOperation();
				pasteMenuItem.setEnabled(false);
				removeThisMenuItem.setEnabled
					(tnc.removeConnectCheck
					 (tnc.getElParent().getElementOperation(),com));
				cutMenuItem.setEnabled(true);
				copyMenuItem.setEnabled(true);
				removeBranchMenuItem.setEnabled(true);
				if (copiedNode != null && copiedNode instanceof TNElement) {
					ElementOperation copiedElement = 
						((TNElement)copiedNode).getElementOperation();
					if (com.canAddElement() && 
						com.acceptOpElement(copiedElement)) {
						pasteMenuItem.setEnabled(true);
					}
				} else if (copiedNode != null && 
						   copiedNode instanceof TNAttribute &&
						   tnc.getObject() instanceof AttributeOperation) {
					pasteMenuItem.setEnabled(true);
				}
			}

			void attributeSelected() {
				cutMenuItem.setEnabled(true);
				copyMenuItem.setEnabled(true);
				pasteMenuItem.setEnabled(false);
				removeThisMenuItem.setEnabled(false);
				removeBranchMenuItem.setEnabled(true);
			}

			public void message(String a,int t,Object selectedNode) {
				undoMenuItem.setEnabled(modulePanel.canUndo());
				if (selectedNode == null ) {
					disableAllMenu();
					return;
				}
				if (selectedNode instanceof TNElement) {
					TNElement tnc = (TNElement)selectedNode;
					if (selectedNode instanceof TNModule || 
						selectedNode instanceof TNHedgeRuleRoot) {
						moduleSelected();
					} else {
						elementSelected(tnc);
					}
				} else if (selectedNode instanceof TNAttribute) {
					attributeSelected();
				}
			}

		};

}

class AboutDialog extends JDialog {
	
	AboutDialog(JFrame frame) {
		super(frame,true);
		initComponents();
		pack();
		SwingUtil.setCenter(this);
	}

	private JPanel buttonPanel;
	private JButton okButton;

	private void initComponents () {
		buttonPanel = new JPanel ();
		okButton = new JButton ();
		addWindowListener (new WindowAdapter () {
				public void windowClosing (WindowEvent evt) {
					dispose();
				}
			});
		okButton.setText (MainApp.getResString("Ok"));
		okButton.addActionListener (new ActionListener () {
				public void actionPerformed (ActionEvent evt) {
					dispose();
				}
			});
		buttonPanel.add (okButton);
		getContentPane ().add (buttonPanel, BorderLayout.SOUTH);

		JPanel panel = new JPanel(new BorderLayout());
		Shader sh1 = new Shader(MainApp.getResString("programName"),
								0xcccccc,// shade color
								0xffffff,// back ground color
								0x449955,// title color
								"Serif",// font name
								35,// font size
								Shader.BOLD); // font style
		Shader sh2 = new Shader("version "+MainApp.getResString("version"),
								0xcccccc,// shade color
								0xffffff,// back ground color
								0x8899ff,// title color
								9,5,5,
								"Serif",// font name
								20,// font size
								Shader.BOLD); // font style
		panel.add(sh1,BorderLayout.NORTH);
		panel.add(sh2,BorderLayout.CENTER);
		JPanel infos = new JPanel(new GridLayout(2,1,5,5));
		infos.add(new JLabel(MainApp.getResString("copyright")));
		infos.add(new JLabel(MainApp.getResString("email")));
		//infos.add(new JLabel(MainApp.getResString("webpage")));
		panel.add(infos,BorderLayout.SOUTH);
		panel.setBorder(new EtchedBorder());
		getContentPane ().add (panel, BorderLayout.CENTER);
	}
	
}
