/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.text.JTextComponent;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import java.util.Stack;
import java.util.List;
import jp.gr.java_conf.ccs2.comp.MessageListener;
import jp.gr.java_conf.ccs2.tool.mkrelax.*;

public class ModulePanel extends JPanel {

	private ModuleModel module;
	private JTabbedPane tab;

	//undo
	private Stack undoStack = new Stack();
	private boolean enableUndo = true;
	private int undoSize = 4;

	//Object Tree area
	private ObjectTreeModel objectTreeModel;
	private JTree objectTree;
	private JPanel propertyPanel;

	//Attribute Pool Tree area
	private AttributePoolTreeModel poolTreeModel;
	private JTree poolTree;
	private JPanel poolPropertyPanel;

	//Hedge Role Tree area
	private HedgeRuleTreeModel hedgeTreeModel;
	private JTree hedgeTree;
	private JPanel hedgePropertyPanel;


	//source area
	private JTextArea sourceText = new JTextArea();

	//selection update
	private MessageListener compListener;

	public ModulePanel(ModuleModel mm) {
		super(new BorderLayout());
		enableUndo = !MainApp.getAppContext().getConfig().getOption("noundo");
		undoSize = MainApp.getAppContext().getConfig().getOptionInteger("undoNum")+1;
		if (undoSize<2) {
			enableUndo = false;
			undoSize=0;
		}
		module = mm;
		initGui();
	}

	public String getModuleName() {
		return module.getModuleName();
	}
	public String getModuleFilename() {
		return module.getModuleFilename();
	}
	public boolean isModuleModified() {
		return module.isModified();
	}

	public void setCompListener(MessageListener ss) {
		compListener = ss;
	}

	public void newModule() {
		module = ModuleModel.createNewModule();
		undoStack.clear();
		reinitGui();
	}

	public void saveModule(String filename) {
		try {
			module.setModuleFilename(filename);
			ModuleModel.saveModuleWithRelax(module,filename);
			reinitTreeModel();
		} catch (IOException e) {
			JOptionPane.showMessageDialog
				(this,e.getMessage(),
				 MainApp.getResString("ErrorDialog"),
				 JOptionPane.ERROR_MESSAGE);
		}
	}

	public void loadModule(String filename) {
		try {
			module = ModuleModel.loadModuleFromRelax(filename);
			undoStack.clear();
			reinitGui();
		} catch (IOException e) {
			JOptionPane.showMessageDialog
				(this,e.getMessage(),
				 MainApp.getResString("ErrorDialog"),
				 JOptionPane.ERROR_MESSAGE);
		}
	}

	protected void reloadModule(List reloadEntries) {
		try {
			module = RelaxReader.getModuleWithReloading(module,reloadEntries);
			reinitGui();
		} catch (MkRelaxException e) {
			JOptionPane.showMessageDialog
				(this,e.getMessage(),
				 MainApp.getResString("ErrorDialog"),
				 JOptionPane.ERROR_MESSAGE);
		}
	}

	public boolean canUndo() {
		return (undoStack.size() > 1) ? true : false;
	}

	protected void prepareUndo() {
		if (!enableUndo) return;
		String obj = module.getRelax();
		if (undoStack.size() > 0 && 
			obj.equals((String)undoStack.peek())) return;
		if (!undoStack.isEmpty()) {
			module.modified();
		}
		undoStack.push(obj);
		while (undoStack.size() > undoSize) {
			undoStack.remove(0);
		}
		MainApp.getAppContext().getMonitor().debug
			("UNDO:"+undoStack.size());
	}

	public void undo() {
		if (undoStack.size()>1) {
			try {
				undoStack.pop();
				module = ModuleModel.relax2module((String)undoStack.pop(),
												  module.getModuleFilename());
			} catch (MkRelaxException e) {
				JOptionPane.showMessageDialog
					(this,e.getMessage(),
					 MainApp.getResString("ErrorDialog"),
					 JOptionPane.ERROR_MESSAGE);
			}
			reinitTreeModel();
		}
	}

	private void initGui() {
		prepareUndo();
		tab = new JTabbedPane();
		tab.add(MainApp.getResString("objectPane"),
				initObjectPane());
		tab.add(MainApp.getResString("attributePoolPane"),
				initPoolPane());
		tab.add(MainApp.getResString("hedgeRulePane"),
				initHedgeRulePane());
		tab.add(MainApp.getResString("sourcePane"),
				initSourcePane());
		tab.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					if (tab.getSelectedIndex() == 3) {// WATCH IT
						updateSourceText();
					}
				}
			});
		this.add(tab,BorderLayout.CENTER);
		getCurrentTree().setSelectionRow(0);
		getCurrentTree().requestFocus();
	}

	JComponent initObjectPane() {
		objectTree = new JTree();
		initObjectTreeModel();
		return initTree(objectTree,new ObjectTreeRenderer(),
						initObjectPropertyPane());
	}

	Component initPoolPane() {
		poolTree = new JTree();
		initPoolTreeModel();
		return initTree(poolTree,new ObjectTreeRenderer(),initPoolPropertyPane());
	}

	Component initHedgeRulePane() {
		hedgeTree = new JTree();
		initHedgeTreeModel();
		return initTree(hedgeTree,new ObjectTreeRenderer(),initHedgePropertyPane());
	}

	JComponent initTree(JTree tree,TreeCellRenderer renderer,
						JComponent propPanel) {
		tree.setCellRenderer(renderer);
		tree.getSelectionModel().setSelectionMode
			(TreeSelectionModel.SINGLE_TREE_SELECTION);
		tree.putClientProperty("JTree.lineStyle", "Angled");
		tree.addTreeSelectionListener(treeValueChangedListener);
		tree.addMouseListener(treeMouseAdapter);
		tree.addKeyListener(keyAdapterOnTree);
		expandAllTrees(tree);
		JScrollPane sc = new JScrollPane(tree);

		Dimension dim = new Dimension(320,400);
		sc.setPreferredSize(dim);
		JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		splitPane.setLeftComponent(sc);
		splitPane.setRightComponent(propPanel);
		splitPane.setContinuousLayout(false);
		return splitPane;
	}

	void expandAllTrees(JTree jt) {
		if (jt != null) {
			for( int i = 0; true; i++ ){
				TreePath path = jt.getPathForRow(i);
				if(path == null) break;
				jt.expandPath(path);
			}
		}
	}

	void reinitGui() {
		reinitTreeModel();
	}

	void reinitTreeModel() {
		prepareUndo();
		int [] ret = getCurrentTree().getSelectionRows();
		int ps = 0;
		if (ret != null && ret.length != 0) ps = ret[0];
		initObjectTreeModel();
		initPoolTreeModel();
		initHedgeTreeModel();
		expandAllTrees(objectTree);
		expandAllTrees(poolTree);
		expandAllTrees(hedgeTree);
		getCurrentTree().requestFocus();
		getCurrentTree().setSelectionRow(ps);
	}

	void initObjectTreeModel() {
		objectTreeModel = new ObjectTreeModel(module,messageListener);
		objectTree.setModel(objectTreeModel);
	}

	void initPoolTreeModel() {
		poolTreeModel = new AttributePoolTreeModel(module,messageListener);
		poolTree.setModel(poolTreeModel);
	}

	void initHedgeTreeModel() {
		hedgeTreeModel = new HedgeRuleTreeModel(module,messageListener);
		hedgeTree.setModel(hedgeTreeModel);
	}

	JComponent initObjectPropertyPane() {
		propertyPanel = new JPanel();
		return initGeneralPropertyPane("propertyPanel",propertyPanel);
	}

	JComponent initPoolPropertyPane() {
		poolPropertyPanel = new JPanel();
		return initGeneralPropertyPane("propertyPanel",poolPropertyPanel);
	}

	JComponent initHedgePropertyPane() {
		hedgePropertyPanel = new JPanel();
		return initGeneralPropertyPane("propertyPanel",hedgePropertyPanel);
	}

	JComponent initGeneralPropertyPane(String res,JPanel panel) {
		panel.setLayout(new BorderLayout());
		panel.setBorder
			(new TitledBorder(MainApp.getResString(res)));
		JScrollPane sc = new JScrollPane(panel);
		sc.setPreferredSize(new Dimension(280,400));
		return sc;
	}

	TreeSelectionListener treeValueChangedListener = new TreeSelectionListener() {
			public void valueChanged(TreeSelectionEvent e) {
				if (compListener != null) {
					compListener.message("",0,getSelectedNode());
				}
				showObjectProperty();
			}
		};

	MouseAdapter treeMouseAdapter = new MouseAdapter() {
			boolean showed = false;
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger() || e.isControlDown()) {
					execPopup(e);
				}
			}

			public void mouseReleased(MouseEvent e) {
				if ((e.isPopupTrigger() || e.isControlDown()) && !showed) {
					execPopup(e);
				}
				showed = false;
			}

			void execPopup(MouseEvent e) {
				TreePath path = getCurrentTree().getClosestPathForLocation
					(e.getX(), e.getY());
				TNCommon com = translatePath(path);
				if (com == null) return;
				getCurrentTree().setSelectionPath(path);
				int xx = e.getX();
				int yy = e.getY();
				showPopupMenu(e.getComponent(),xx,yy,com);
			}
		};

	Component initSourcePane() {
		sourceText.setEditable(false);
		sourceText.setLineWrap(true);
		sourceText.addKeyListener(keyAdapterOnSource);
		return new JScrollPane(sourceText);
	}

	void updateSourceText() {
		sourceText.setText(ModuleModel.module2relax(module));
	}

	public TNCommon getSelectedNode() {
		TreePath path = getCurrentTree().getSelectionPath();
		return translatePath(path);
	}

	public void showPopupMenu() {
		TNCommon tnc = getSelectedNode();
		if (tnc == null) return;
		Rectangle r = getCurrentTree().getPathBounds
			(getCurrentTree().getSelectionPath());
		if (r == null) return;
		showPopupMenu(getCurrentTree(),r.x+r.width,r.y+r.height,tnc);
	}

	public void showPopupMenu(Component com,int x,int y,TNCommon tnc) {
		JPopupMenu pop = tnc.getPopupMenu();
		if (pop == null) return;
		int yy = y;
		//Point ps = com.getLocationOnScreen();
		Point ps = com.getLocation();
		Dimension scSize = Toolkit.getDefaultToolkit().getScreenSize();
		MainApp.getAppContext().getMonitor().debug("SCREEN: ps="+ps);
		MainApp.getAppContext().getMonitor().debug("SCREEN: sc="+scSize);
		if ( (scSize.height/2) < ps.y ) {
			yy=y-pop.getSize().height;
		}
		pop.show(com,x,yy);
		pop.requestFocus();
		pop.setNextFocusableComponent(getCurrentTree());
	}

	JTree getCurrentTree() {
		switch (tab.getSelectedIndex()) {
		case 1:
			return poolTree;
		case 2:
			return hedgeTree;
		default:
			return objectTree;
		}
	}
	JPanel getCurrentPanel() {
		switch (tab.getSelectedIndex()) {
		case 1:
			return poolPropertyPanel;
		case 2:
			return hedgePropertyPanel;
		default:
			return propertyPanel;
		}
	}
	TreeModel getCurrentTreeModel() {
		return getCurrentTree().getModel();
	}

	TNCommon translatePath(TreePath path) {
		if (path == null) return null;
		Object obj = path.getLastPathComponent();
		if (obj == null) return null;
		if (obj instanceof TNCommon) {
			getCurrentPanel().removeAll();
			TNCommon com = (TNCommon)obj;
			return com;
		}
		return null;
	}

	void showObjectProperty() {
		TNCommon com = getSelectedNode();
		if (com == null) {
			getCurrentPanel().revalidate();
			getCurrentPanel().repaint();
			return;
		}
		getCurrentPanel().removeAll();
		JPanel ppp = com.getPropertyPaneGen(keyAdapterOnPropPanel);
		ppp.setOpaque(true);
		getCurrentPanel().add(ppp,BorderLayout.CENTER);
		getCurrentPanel().revalidate();
		getCurrentPanel().repaint();
	}

	void disablePanel(Container comp) {
		Component [] coms = comp.getComponents();
		for (int i=0;i<coms.length;i++) {
			if (coms[i] == null) continue;
			if (coms[i] instanceof Container) 
				disablePanel((Container)coms[i]);
			if (coms[i] instanceof JTextComponent ||
				coms[i] instanceof AbstractButton) {
				coms[i].setEnabled(false);
			}
		}
	}

	void forceKeyEvent(Container comp) {
		Component [] coms = comp.getComponents();
		for (int i=0;i<coms.length;i++) {
			if (coms[i] == null) continue;
			if (coms[i] instanceof Container) 
				forceKeyEvent((Container)coms[i]);
			if (coms[i] instanceof JTextComponent ||
				coms[i] instanceof AbstractButton) {
				coms[i].removeKeyListener(keyAdapterOnPropPanel);
				coms[i].addKeyListener(keyAdapterOnPropPanel);
			}
		}
	}

	void processTabKey(int e) {
		switch (e) {
		case KeyEvent.VK_F1:
			tab.setSelectedIndex(0);
			objectTree.requestFocus();
			break;
		case KeyEvent.VK_F2:
			tab.setSelectedIndex(1);
			poolTree.requestFocus();
			break;
		case KeyEvent.VK_F3:
			tab.setSelectedIndex(2);
			hedgeTree.requestFocus();
			break;
		case KeyEvent.VK_F4:
			tab.setSelectedIndex(3);
			sourceText.requestFocus();
			break;
		default:
		}
	}

	KeyAdapter keyAdapterOnPropPanel = new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				switch (e.getKeyCode()) {
				case KeyEvent.VK_F12:
					getCurrentTree().requestFocus();
					return;
				default:
				}
				processTabKey(e.getKeyCode());
			}
		};

	KeyAdapter keyAdapterOnSource = new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				processTabKey(e.getKeyCode());
			}
		};

	void traverseTab(JComponent comp) {
		if (!comp.isFocusTraversable()) {
			FocusManager.getCurrentManager().focusNextComponent(comp);
		}
	}

	private KeyAdapter keyAdapterOnTree = new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				switch (e.getKeyCode()) {
				case KeyEvent.VK_F12:
					traverseTab(getCurrentPanel());
					return;
				case KeyEvent.VK_ESCAPE:
					showPopupMenu();
					return;
				default:
				}
				processTabKey(e.getKeyCode());
			}
		};

	private MessageListener messageListener = new MessageListener() {
			public void message(String n,int t,Object h) {
				switch (t) {
				case REPAINT:
					if (h==null) {
						h=getCurrentTreeModel().getRoot();
					}
					((DefaultTreeModel)getCurrentTreeModel()).nodeChanged((TNCommon)h);
					prepareUndo();
					return;
				case DISPOSE:
					reinitTreeModel();
					return;
				case RECALC:
					reloadModule((List)h);
					return;
				default:
				}
				throw new InternalError("No such message. ["+t+"]");
			}
		};

}
