/* MkRelax, Visual Relax Editor
 *	Copyright (C) 2001-2002 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.tool.mkrelax.relaxer;
import jp.gr.java_conf.ccs2.tool.mkrelax.*;
import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.List;
import javax.swing.*;
import jp.gr.java_conf.ccs2.comp.SwingUtil;
import jp.gr.java_conf.ccs2.util.StringUtil;
import jp.gr.java_conf.ccs2.util.ExtensionFileFilter;
import jp.gr.java_conf.ccs2.util.ResourceFile;
import jp.gr.java_conf.ccs2.io.EasyWriter;
import java.net.URL;
import java.util.Locale;
import java.text.MessageFormat;
import javax.swing.border.TitledBorder;
import java.io.File;
import java.io.IOException;
import java.io.BufferedReader;
import java.io.InputStreamReader;


public class RelaxerDialog extends JFrame {

	private static final String TITLE_HINT = "{0} ({1})";

	private String relaxPath;
	private SwitchEntry [] switches;
	private OptionEntry [] options;
	private Entry [] totalEntries;

	private Locale locale = Locale.getDefault();

	private JTextArea outputArea;
	private JTextField pathField;
	private JButton execButton,cancelButton;
	private boolean show = false;

	private Process process;

	public RelaxerDialog(String relaxPath) {
		super(MainApp.getResString("Relaxer.DialogTitle"));
		this.relaxPath = relaxPath;
		setupRelaxerResource();
		setupGui();
		SwingUtil.setCenter(this);
	}

	public boolean isShown() {
		return show;
	}
	public void setPath(String p) {
		relaxPath = p;
	}

	public void show() {
		show = true;
		super.show();
	}

	public void dispose() {
		show = false;
		super.dispose();
	}
	
	private ResourceFile currentResource;
	private void loadCurrentResource() {
		File prop = new File(new File(relaxPath).getParent(),
							 "Relaxer.properties");
		if (prop.exists()) {
			currentResource = ResourceFile.getResourceFile(prop.getPath());
		}
	}


	private void setupRelaxerResource() {
		loadCurrentResource();
		URL url = "".getClass().getResource
			("/jp/gr/java_conf/ccs2/tool/mkrelax/relaxer/relaxerOptions.xml");
		RelaxerOptions relaxerOptions = null;
		try {
			relaxerOptions = new RelaxerOptions(url);
		} catch (Exception e) {
			String [] args = {e.getMessage(),e.getClass().getName()};
			String ret = MessageFormat.format
				(MainApp.getResString("InternalError{0}{1}"),args);
			MainApp.getAppContext().getMonitor().recordStackTrace(e);
			throw new InternalError(ret);
		}
		totalEntries = new Entry[relaxerOptions.getSwitchCount()+relaxerOptions.getOptionCount()];
		Switch [] sws = relaxerOptions.getSwitch();
		switches = new SwitchEntry[sws.length];
		for (int i=0;i<sws.length;i++) {
			switches[i] = new SwitchEntry
				(sws[i].getName(),sws[i].getDefault(),
				 getDescription(sws[i].getDescription()),
				 (sws[i].checkForce() && sws[i].getForce()));
			totalEntries[i] = switches[i];
		}
		Option [] ops = relaxerOptions.getOption();
		options = new OptionEntry[ops.length];
		for (int i=0;i<ops.length;i++) {
			options[i] = new OptionEntry
				(ops[i].getName(),ops[i].getDefault(),
				 getDescription(ops[i].getDescription()),
				 (ops[i].checkForce() && ops[i].getForce()));
			totalEntries[i+switches.length] = options[i];
		}
	}

	private String getCurrentOption(String name) {
		if (currentResource == null) return null;
		try {
			return (String)(currentResource.getString(name));
		} catch (RuntimeException e) {
		}
		return null;
	}

	private String getDescription(Description [] desc) {
		if (desc == null || desc.length == 0) 
			return MainApp.getResString("Relaxer.NoDescription");
		String def=MainApp.getResString("Relaxer.NoDescription");
		String local=null;
		for (int i=0;i<desc.length;i++) {
			if (StringUtil.isNull(desc[i].getLang())) {
				def = desc[i].getContent();
				continue;
			}
			if (locale.getLanguage().equals
				(new Locale(desc[i].getLang(), "", "").getLanguage())) {
				local = desc[i].getContent();
			}
		}
		return StringUtil.isNull(local) ? def : local;
	}
	
	private void setupGui() {
		JTabbedPane tab = new JTabbedPane();
		tab.add(MainApp.getResString("Relaxer.optionsTabTitle"),
				initOptionsPane());
		tab.add(MainApp.getResString("Relaxer.switchesTabTitle"),
				initSwitchesPane());
		tab.add(MainApp.getResString("Relaxer.execTabTitle"),
				initRelaxerPane());
		getContentPane().add(tab,BorderLayout.CENTER);
		getContentPane().add(initExitButton(),BorderLayout.SOUTH);
		setSize(550,500);
	}

	private JPanel initExitButton() {
		JPanel panel = new JPanel();
		JButton ex = new JButton(MainApp.getResString("Relaxer.exitButtonTitle"));
		ex.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dispose();
				}
			});
		panel.add(ex);
		JButton reset = new JButton(MainApp.getResString("Relaxer.resetAllButtonTitle"));
		reset.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					resetAllOptions();
				}
			});
		panel.add(reset);
		return panel;
	}

	private void resetAllOptions() {
		for (int i=0;i<options.length;i++) {
			options[i].reset();
		}
		for (int i=0;i<switches.length;i++) {
			switches[i].reset();
		}
	}

	private String formatTitle(String t,String h) {
		return MessageFormat.format(TITLE_HINT,new String[] {t,h});
	}

	private JComponent initOptionsPane() {
		JPanel panel = new JPanel(new GridLayout(options.length,1));
		for (int i=0;i<options.length;i++) {
			final int ii = i;
			JPanel pp = new JPanel();
			pp.setBorder(new TitledBorder
				(formatTitle(options[i].getName(),
							 options[i].getDescription())));
			final JTextField field = (JTextField)options[i].getComponent();
			pp.add(field);
			JButton res = new JButton
				(MainApp.getResString("Relaxer.resetButtonTitle"));
			res.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						options[ii].reset();
					}
				});
			pp.add(res);
			options[i].setComponent(field);
			panel.add(pp);
		}
		return new JScrollPane(panel);
	}

	private JComponent initSwitchesPane() {
		JPanel panel = new JPanel(new GridLayout(switches.length/2,2));
		for (int i=0;i<switches.length;i++) {
			final int ii = i;
			JPanel pp = new JPanel();
			pp.setBorder(new TitledBorder
				(formatTitle(switches[i].getName(),
							 switches[i].getDescription())));
			final JCheckBox check = (JCheckBox)switches[i].getComponent();
			pp.add(check);
			JButton res = new JButton
				(MainApp.getResString("Relaxer.resetButtonTitle"));
			res.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						switches[ii].reset();
					}
				});
			pp.add(res);
			switches[i].setComponent(check);
			panel.add(pp);
		}
		return new JScrollPane(panel);
	}

	private JComponent initRelaxerPane() {
		JPanel panel = new JPanel(new BorderLayout());
		panel.add(initRelaxerPaneTopPanel(),BorderLayout.NORTH);
		panel.add(initRelaxerPaneBottomPanel(),BorderLayout.CENTER);
		return panel;
	}

	private JComponent initRelaxerPaneTopPanel() {
		JPanel topPanel = new JPanel(new BorderLayout());
		topPanel.setBorder(new TitledBorder(MainApp.getResString("Relaxer.execRelaxerTitle")));
		JPanel line = new JPanel();
		line.add(new JLabel(MainApp.getResString("Relaxer.path")));
		pathField = new JTextField(24);
		pathField.setText(getRelaxerPath());
		line.add(pathField);
		JButton pathSel = new JButton(MainApp.getResString("Relaxer.selectPathButtonTitle"));
		pathSel.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					JFileChooser fc = new JFileChooser("/");
					int returnVal = fc.showOpenDialog(RelaxerDialog.this);
					if (returnVal == JFileChooser.APPROVE_OPTION) {
						File file = fc.getSelectedFile();
						String filename = file.getPath();
						pathField.setText(filename);
					}
				}
			});
		line.add(pathSel);
		topPanel.add(line,BorderLayout.CENTER);
		//
		line = new JPanel();

		JButton genButton = new JButton(MainApp.getResString("Relaxer.generateButtonTitle"));
		genButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					generatePropertyFile();
				}
			});
		execButton = new JButton(MainApp.getResString("Relaxer.execButtonTitle"));
		execButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					execRelaxer();
				}
			});
		cancelButton = new JButton(MainApp.getResString("Relaxer.cancelButtonTitle"));
		cancelButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					cancelRelaxer();
				}
			});
		cancelButton.setEnabled(false);
		line.add(genButton);
		line.add(execButton);
		line.add(cancelButton);
		topPanel.add(line,BorderLayout.SOUTH);
		return topPanel;
	}

	private String getRelaxerPath() {
		String relaxerPath = MainApp.getAppContext().getConfig().getOptionString("relaxerPath");
		if (System.getProperty("file.separator").equals("\\")) {
			if (relaxerPath.indexOf(".bat")==-1)
				relaxerPath = relaxerPath+".bat";
			if (relaxerPath.indexOf(":") == -1)
				relaxerPath = "c:"+relaxerPath;
		}
		return relaxerPath;
	}

	private JComponent initRelaxerPaneBottomPanel() {
		JPanel panel = new JPanel(new BorderLayout());
		panel.setBorder(new TitledBorder(MainApp.getResString("Relaxer.outputPanelTitle")));
		outputArea = new JTextArea(8,40);
		panel.add(new JScrollPane(outputArea));
		return panel;
	}

	private void generatePropertyFile() {
		JFileChooser fc = new JFileChooser(System.getProperty("user.dir"));
		fc.setSelectedFile(new File("Relaxer.properties"));
		ExtensionFileFilter ff = new ExtensionFileFilter(".properties");
		ff.setDescription(MainApp.getResString("Relaxer.propertiesDescription"));
		fc.setFileFilter(ff);
		int returnVal = fc.showSaveDialog(RelaxerDialog.this);
		if (returnVal != JFileChooser.APPROVE_OPTION) return;
		File file = fc.getSelectedFile();
		String filename = file.getPath();
		EasyWriter out = new EasyWriter(filename);
		if (!out.isOk()) {
			JOptionPane.showMessageDialog
				(this,MainApp.getResString("IOError{0}",filename),
				 MainApp.getResString("ErrorDialog"),
				 JOptionPane.ERROR_MESSAGE);
			return;
		}
		for (int i=0;i<totalEntries.length;i++) {
			String a = totalEntries[i].propertyExpression();
			if (StringUtil.isNull(a)) continue;
			out.println(a);
		}
		out.flush();
		out.close();
	}

	private String [] setupArguments() {
		List argList = new ArrayList();
		argList.add(pathField.getText());
		for (int i=0;i<totalEntries.length;i++) {
			String a = totalEntries[i].consoleExpression();
			if (StringUtil.isNull(a)) continue;
			argList.add(a);
		}
		argList.add(relaxPath);
		String [] ret = new String[argList.size()];
		ret = (String[])argList.toArray(ret);
		//
		StringBuffer sb = new StringBuffer();
		for (int i=0;i<ret.length;i++) {
			sb.append(ret[i]).append(" ");
		}
		appendEvent(sb.toString());
		MainApp.getAppContext().getMonitor().debug
			("RELAXER_ARG:"+sb.toString());
		return ret;
	}

	private void execRelaxer() {
		Thread thread = new Thread(new Runnable() {
				public void run() {
					runRelaxer();
				}
			});
		thread.start();
	}

	private boolean stdReady=false;
	private boolean errReady=false;
	private void runRelaxer() {
		try {
			execButton.setEnabled(false);
			cancelButton.setEnabled(true);
			process = Runtime.getRuntime().exec(setupArguments());
			stdReady = true;
			errReady = true;
			Thread thread1 = new Thread(new Runnable() {
					public void run() {
						readStdout(new BufferedReader
							(new InputStreamReader(process.getInputStream())));
					}});
			thread1.start();
			Thread thread2 = new Thread(new Runnable() {
					public void run() {
						readStderr(new BufferedReader
							(new InputStreamReader(process.getErrorStream())));
					}});
			thread2.start();
			while(stdReady || errReady) {
				Thread.sleep(600);
				MainApp.getAppContext().getMonitor().debug
					("RELAXER_OUT:(loop) "+stdReady+":"+errReady);
			}
		} catch (IOException e) {
			JOptionPane.showMessageDialog
				(RelaxerDialog.this,MainApp.getResString("Relaxer.IOException{0}",
														 e.getMessage()),
				 MainApp.getResString("ErrorDialog"),
				 JOptionPane.ERROR_MESSAGE);
			MainApp.getAppContext().getMonitor().warning
				(MainApp.getResString("Relaxer.IOException{0}",e.getMessage()));
		} catch (InterruptedException e) {
			MainApp.getAppContext().getMonitor().recordStackTrace(e);
		} finally {
			stdReady = false;
			errReady = false;
			execButton.setEnabled(true);
			cancelButton.setEnabled(false);
			process = null;
			appendEvent("\n"+MainApp.getResString("Relaxer.processFinished"));
		}
	}

	private void readStdout(BufferedReader in) {
		try {
			while(true) {
				String line = in.readLine();
				if (line == null) break;
				MainApp.getAppContext().getMonitor().debug
					("RELAXER_OUTPUT:"+line);
				appendEvent(line);
			}
		} catch (IOException e) {
			final IOException ee = e;;
			Runnable task = new Runnable() {
					public void run() {
						JOptionPane.showMessageDialog
							(RelaxerDialog.this,
							 MainApp.getResString("Relaxer.IOException{0}",
												  ee.getMessage()),
							 MainApp.getResString("ErrorDialog"),
							 JOptionPane.ERROR_MESSAGE);
					}
				};
			SwingUtilities.invokeLater(task);
			MainApp.getAppContext().getMonitor().warning
				(MainApp.getResString("Relaxer.IOException{0}",
									  e.getMessage()));
		} finally {
			stdReady=false;
		}
	}

	private void readStderr(BufferedReader in) {
		try {
			while(true) {
				String line = in.readLine();
				if (line == null) break;
				MainApp.getAppContext().getMonitor().debug
					("RELAXER_OUTPUT:(err)"+line);
				appendEvent(line);
			}
		} catch (IOException e) {
			MainApp.getAppContext().getMonitor().warning
				(MainApp.getResString("Relaxer.IOException{0}",
									  e.getMessage()));
		} finally {
			errReady=false;
		}
	}

	private void cancelRelaxer() {
		process.destroy();
	}

	private void appendEvent(String line) {
		final String aline = line;
		Runnable task = new Runnable() {
				public void run() {
					outputArea.append(aline);
					outputArea.append("\n");
				}
			};
		SwingUtilities.invokeLater(task);
	}

	abstract class Entry {
		String name,description;
		JComponent component;
		boolean force = false;
		Entry(String n,String d,JComponent com,boolean f) {
			name = n;
			description = d;
			component = com;
			force = f;
		}
		String getName() {
			return name;
		}
		String getDescription() {
			return description;
		}
		void setComponent(JComponent com) {
			component = com;
		}
		JComponent getComponent() {
			return component;
		}
		boolean isForced() {return force;}
		abstract void reset();
		abstract boolean isSame();
		abstract String consoleExpression();
		abstract String propertyExpression();
	}

	class SwitchEntry extends Entry{
		boolean defaultState;
	
		SwitchEntry(String n,boolean d,String doc,boolean f) {
			super(n,doc,new JCheckBox(),f);
			defaultState = d;
			setState(d);
			if (getCurrentOption(n)!=null) {
				if (getCurrentOption(n).equals("true")) setState(true);
				else setState(false);
			} 
		}
		void reset() {
			setState(defaultState);
		}
		boolean getState() {
			return ((JCheckBox)component).isSelected();
		}
		void setState(boolean b) {
			((JCheckBox)component).setSelected(b);
		}
		boolean isSame() {
			return (getState() == defaultState);
		}
		String consoleExpression() {
			if (isForced() && getState()) 
				return "-"+this.getName();
			if (isSame() || !getState()) return null;
			return "-"+this.getName();
		}
		String propertyExpression() {
			if (isSame() && !isForced()) return null;
			return this.getName()+"="+(getState() ? "true":"false");
		}
	}

	class OptionEntry extends Entry {
		String name,defaultContent,curContent;
		OptionEntry(String n,String d,String doc,boolean f) {
			super(n,doc,new JTextField(30),f);
			defaultContent = d;
			setContent(defaultContent);
			if (getCurrentOption(n)!=null) {
				setContent(getCurrentOption(n));
			} 
		}
		void reset() {
			setContent(defaultContent);
		}
		String getContent() {
			return ((JTextField)component).getText();
		}
		void setContent(String s) {
			((JTextField)component).setText(s);
		}
		boolean isSame() {
			return (defaultContent.equals(getContent()));
		}
		String consoleExpression() {
			if (isSame() && !isForced()) return null;
			return "-"+this.getName()+":"+getContent();
		}
		String propertyExpression() {
			if (isSame() && !isForced()) return null;
			return this.getName()+"="+getContent();
		}
	}


}
