require "xmlrpc/client"
require "xmlrpc/server"
require "singleton"
require "thread"
require "socket"

# Author::    Masashi Sakurai (mailto:m.sakurai@dream.com)
# Copyright:: Copyright (C) 2005 Masashi Sakurai 
# License::   Distributes under the same terms as Ruby, except javassist.jar (MPL1.1).

#=JavaBridge
#
# This module provides a communication facility between Ruby and Java.
# The current implementation is written by pure Java and pure ruby.
# You can feel ruby syntax and gain Java power.
#
#== Simple example and features
#
#=== Hello world
#
#  require 'yajb/jbridge'
#  include JavaBridge
#  jimport "javax.swing.*"
#  
#  jstatic(:JOptionPane).showMessageDialog( jnew(:JFrame), "Hello World!")
#  # this Java window sometimes shows behind the other window.
#
#=== GUI
#
# You can use Java GUI components as you use on Java.
# See sample scripts:
#
#* gui_simple.rb    : Simple GUI (Implementation of java.awt.event interfaces)
#* gui_dialog.rb    : JDialog Demo (Handling the Java-AWT thread)
#* gui_table.rb     : JTable Demo (Implementation of MVC)
#* gui_graphics.rb  : Drawing graphics (Implementing JComponent and drawing graphics)
#
# Writing scripts in UTF-8, you can use multibyte code correctly.
#
#=== POI
#
#* ext_poi.rb : POI Demo (Using external classpath and multibyte handling)
#
# POI sample demonstrates multibyte handling.
#
#=== Meta operation
#
# This package contains the javassist, the byte engineering tool library.
# You can make the POJO implementation from scratch.
# See website, http://www.csg.is.titech.ac.jp/~chiba/javassist/index.html
#
#=== Current state
#
# This implementation:
#
#* can not handle Java's exception.
#* have not implemented Ruby-Java GC.
#* can not show a message from JVM into ruby's standerd output.
#* use two TCP ports.
#* very slow, because of the XMLRPC communication.
#
#== Configuration 
#
# When you call a communication method for the first time, 
# this module starts up JVM and constructs the bridge between Ruby and Java.
# You can configure JVM and Bridge parameters via hash object, as follows:
#
# JBRIDGE_OPTIONS = {
#   :classpath => "$CLASSPATH",   # this record will be evaluated as `echo #{classpath}`.
#   :jvm_path =>  "java",         # if you need, set fullpath to the JVM.
#                                 # if nil, the jbridge will not start the JVM by itself.
#   :jvm_vm_args => "",           # VM argument. ex: "-Xmx128m"
#   :jvm_log_file => nil,         # output from JVM. ex: "log.txt"
#   :jvm_log_level => "normal",   # JVM debug level. [debug,verbose,normal,warning,error]
#   :bridge_log => false,         # Ruby debug output. true or false
# 
#   :bridge_driver => JavaBridge::XMLRPCBridge.instance, # defualt implementation
#   :xmlrpc_bridge_port_r2j => 9010,  # communication port: ruby -> java
#   :xmlrpc_bridge_port_j2r => 9009,  # communication port: java -> ruby
#   :xmlrpc_bridge_opened => JavaBridge::XMLRPCBridge::OP_KILL,
#                    # If the port is using, how the program deals the port:
#                    # OP_KILL  : try to kill the previous port and open new port
#                    # OP_ABORT : abort program
#                    # OP_REUSE : use the previous port
# }
#
# When your Ruby program finishes (that is the termination of main thread),
# the communication bridge is also killed.
# If you want to wait for the message from the JVM, such as GUI events,
# you can call the "joinBridge" method to hold the bridge communication.
#
#== TODO
#
#===solving the GC problem (created instances aren't deleted.)
#
#===show stdout from JVM
#
#*improving performance
#*implementation of another driver (pipe, JNI...)
#*changing XML parser
#
#===implementation of another language: scheme, python...
#
#== The lower protocol for XMLRPC
#
#=== Type and Value
#
#* Z:boolean, B:byte, C:char, S:short, I:int, J:long, F:float, String
#* The other types are managed by the object manager with proxy ID.
#
#=== Java
#
#* [proxy ID] = new([fqcn],[values])
#* [proxy ID] = static([class name])
#
#* [proxy ID] = extends([fqcns], [values])
#* impl([proxy ID],[method name], [override flag])
#
#* [fqcn] = classname([proxy ID])
#
#* [value] = ref([proxy ID],[field name])
#* set([proxy ID],[field name],[value])
#
#* [value] = call([proxy ID],[method name],[values])
#* [value] = callsuper([proxy ID],[method name],[values])
#
#* [value] = sessionCall([message],[arguments])
#
#* unlink([proxy ID])
#
#* exit()
#* dump()
#
#=== Ruby
#
#* [value] = call([session ID],[proxy ID],[values])
#
#== Thread session
#

module JavaBridge

  ##########################################################################
  # Bridge driver area
  ##########################################################################

  private

  # abstract communication driver class
  # 
  class AbstractBridgeConnection
	include Singleton

	# java ${VM args} JavaBridge ${BridgeArgs} -logLevel:*** -logfile:***
	# This method should return "BridgeArgs".
	# "VM args" is made by options. 
	# An option is given by "optProc.call(:some_key)".
	def getBridgeArgs(optProc)
	  raise "Not implemented error."
	end

	# If the communication bridge needs additional classpath for the java implementation,
	# the subclass return the classpath that will be used as a classpath argument of JVM.
	# "libpath" is the full path to the directory including jbridge.rb.
	def getBridgeClasspath(libpath)
	  nil
	end
	
	# Startup the communication server.
	# The JVM is started by the framework.
	def startupServer(optProc,receiverProc)
	  raise "Not implemented error."
	end

	# If the module method "joinBridge" is called, this method
	# is called to join the communication thread.
	def joinServer
	  "Do nothing..."
	end

	# Shutdown the communication server.
	# The subclass should kill the communication server running on ruby.
	# The JVM is killed by the framework.
	def shutdownServer
	  raise "Not implemented error."
	end

	# Send a message to JavaBridge.
	# Through this method, the jbridge module send some messages to Java,
	# such as new, call, ref and set method.
	def sendMessageToJava(method,*args)
	  raise "Not implemented error."
	end

	# This method handles the message that is called by the Java.
	# This method translates the Java messages and calls 
	# receiverProc object: "receiverProc.call(objectId,method,args)".
	def createMessageHandler(receiverProc)
	  raise "Not implemented error."
	end
  end

  # This class is a simple default implementation.
  class XMLRPCBridge < AbstractBridgeConnection

	OP_REUSE = :reuse
	OP_ABORT = :abort
	OP_KILL = :kill

	def getBridgeArgs(optProc)
	  portj2r = optProc.call(:xmlrpc_bridge_port_j2r)
	  portr2j = optProc.call(:xmlrpc_bridge_port_r2j)
	  if check_previous_port(optProc.call(:xmlrpc_bridge_opened),portr2j) then
		return "JavaBridge -remoteport:#{portj2r} -javaport:#{portr2j}"
	  else
		return nil
	  end
	end

	def check_previous_port(way,port)
	  begin
		s = TCPSocket.open("localhost", port)
		s.close
		#someone using the port
		case way
		when OP_REUSE
		  puts "Port:#{port} has been opened. Trying to reuse the port..."
		  return nil
		when OP_KILL
		  puts "Port:#{port} has been opened. Trying to kill the port..."
		  client = XMLRPC::Client.new("localhost", "/RPC2", port)
		  client.call2("jb.exit")
		  return 0
		when OP_ABORT
		  puts "Port:#{port} has been opened. Abort this program."
		  abort
		end
	  rescue Errno::ECONNREFUSED => e
		#no connection
		return 0
	  rescue => e
		puts "Failed to deal with the opened port..."
		abort
	  end
	end

	def getBridgeClasspath(libpath)
	  "#{libpath}/yajb.jar"
	end
	
	def startupServer(optProc,_receiverProc)
	  @clientPort = optProc.call(:xmlrpc_bridge_port_r2j)
	  @clientPool = []
	  @pool_lock = Mutex.new
	  myserver = XMLRPC::Server.new(optProc.call(:xmlrpc_bridge_port_j2r),
									"localhost",2,$stdout,false,
									optProc.call(:debugOut))
	  myserver.add_handler("jb",XMLRPCJBReceiver.new(_receiverProc))
	  myserver.add_handler("server.exit") do
		myserver.shutdown
	  end
	  @server = Thread.new {
		myserver.serve
	  }
	  @shutdown_proc = lambda {
		myserver.shutdown
	  }
	end

	def client_call2(method,*args)
	  client = nil
	  @pool_lock.synchronize {
		if @clientPool.empty? then
		  print_debug("  ##Create XMLRPC_Client")
		  client = XMLRPC::Client.new("localhost", "/RPC2", @clientPort)
		else
		  client = @clientPool.pop
		end
	  }
	  begin
		return client.call2("jb.#{method}",*args)
	  ensure
		@pool_lock.synchronize {
		  @clientPool.push(client)
		}
	  end
	end

	def shutdownServer
	  client_call2("exit")
	  @shutdown_proc.call
	end

	def joinServer
	  @server.join
	end

	def sendMessageToJava(method,*args)
	  ok,ret = client_call2(method,*args)
	  print_debug "## >> #{ok}, #{ret}"
	  if ok
		return ret
	  else
		print_debug "Error Code: #{ret.faultCode}"
		print_debug ret.faultString
		raise ret.faultString
	  end
	end

	class XMLRPCJBReceiver
	  def initialize(rproc)
		@receiverProc = rproc
	  end
	  def call(sid,objId,methodName,*args)
		@receiverProc.call(sid,objId,methodName,*args)
	  end
	end

  end
  
  NULL_SYMBOL = "$$$NULL$$$"

  ##########################################################################
  # Configuration area
  ##########################################################################

  @@defaultOptions = {
	:classpath => "$CLASSPATH",
	:jvm_path =>  "java",
	:jvm_vm_args => "",
	:jvm_log_file => nil,
	:jvm_log_level => "normal",
	:bridge_log => false,

	:bridge_driver => XMLRPCBridge.instance,
	:xmlrpc_bridge_port_r2j => 9010,
	:xmlrpc_bridge_port_j2r => 9009,
	:xmlrpc_bridge_opened => JavaBridge::XMLRPCBridge::OP_KILL,
  }

  @@optProc = lambda do |name|
	return @@options[name] if @@options.key?(name)
	@@defaultOptions[name]
  end

  def __jbopt(name)
	@@optProc.call(name)
  end

  def __cpseparator
	if RUBY_PLATFORM["win"].nil? then
	  csep = ":"
	else
	  csep = ";"
	end
  end

  def __search_libpath
	ret = nil
	catch (:found) {
	  $LOAD_PATH.each do |dir|
		Dir.glob("#{dir}/yajb/jbridge.rb") do |path|
		  ret = path.chomp.gsub(/\/jbridge\.rb$/,'')
		  throw :found
		end
	  end
	}
	ret = "." if ret.nil?
	if !RUBY_PLATFORM["cyg"].nil? then
	  ret = `cygpath -w #{ret}`.gsub(/\\/,"/").chomp
	end
	return ret
  end
  
  ##########################################################################
  # Low level API area
  ##########################################################################

  def __build_bridge
	return if defined?(@@objectTable)

	if defined?(JBRIDGE_OPTIONS) then
	  @@options = JBRIDGE_OPTIONS
	else
	  @@options = Hash.new
	end

	@@objectTable = Hash.new
	@@debugOut = __jbopt(:bridge_log)

	print_debug("JavaBridge: Startup communication bridge...")

	@@driver = __jbopt(:bridge_driver)
	__startup_JVM
	@@driver.startupServer(@@optProc, lambda { |*args| __called_by_java(*args)})
	@@class_repository = JClassRepository.new

	print_debug("JavaBridge: Finished initialized.")
  end

  def __startup_JVM
	#JVM CLASSPATH
	cp = `echo "#{__jbopt(:classpath)}"`.chomp
	cp = "#{cp}#{__cpseparator}#{@@driver.getBridgeClasspath(__search_libpath)}"
	print_debug("CLASSPATH: #{cp}")
	#JVM PATH
	path = __jbopt(:jvm_path)
	return unless path
	#JVM VM ARGS
	vmarg = __jbopt(:jvm_vm_args)
	
	#JVM LOG
	if __jbopt(:jvm_log_file).nil? then
	  logfile = ""
	else
	  logfile = " -logfile:#{__jbopt(:jvm_log_file)} "
	end
	loglevel = __jbopt(:jvm_log_level)

	#DRIVER ARGS
	driverArgs = @@driver.getBridgeArgs(@@optProc)
	return if driverArgs.nil?

	cmd = "#{path} #{vmarg} -classpath \"#{cp}\" #{driverArgs} -logLevel:#{loglevel} #{logfile}"
	print_debug(cmd)

	#EXEC COMMAND AND WAIT FOR INIT
	io = IO.popen(cmd,"r")
	while true
	  line = io.gets
	  puts "JVM: #{line}" if @@debugOut
	  abort "Can not start JVM: \n#{cmd}" if line.nil?
	  break if line =~ /^OK/
	end
	#REGISTER JVM KILLER
	at_exit {
	  begin
		break_bridge
	  rescue
		#ignore?
	  end
	  begin
		Process.kill(:QUIT, io.pid)
	  rescue
		#ignore?
	  end
	}
	print_debug("JVM: Initialized.")
  end

  def __register(obj)
	@@objectTable[obj.__objectId] = obj
	obj
  end
  
  def print_debug(out)
	puts out if @@debugOut
  end

  # Called by java
  def __called_by_java(sid,objId,methodName,*args)
	print_debug "%% Received: #{objId}.#{methodName} : #{args.join(", ")}"
	obj = @@objectTable[objId]
	if obj.nil? then
	  # not found extended object
	  raise "Object: #{objId} not found."
	elsif (obj.respond_to?(methodName))
	  # OK
	  Thread.current[:__jb_session] = sid
	  begin
		ret = obj.__jsend__(methodName,*args)
		ret = NULL_SYMBOL if ret.nil? 
		return ret
	  rescue => ev
		puts "EXCEPTION: #{ev}"
		puts ev.backtrace
		sa = args.join(",")
		puts "sid:#{sid}  object:#{obj.__classname}  method:#{methodName}  args:#{sa}:#{args.size}"
		raise ev
	  ensure
		Thread.current[:__jb_session] = nil
	  end
	else
	  # not found overrided method
	  raise "Method: #{methodName} not found in #{obj.__classname}"
	end
  end

  # calling the JavaBridge API
  def __send_message_to_java(method,*args)
	sid = Thread.current[:__jb_session]
	if sid then
	  # session thread logic
	  print_debug "## SESSION[#{sid}]  SendMessage: #{method}( #{args.join(", ")} )"
	  args.insert(0,sid,method)
	  return @@driver.sendMessageToJava("sessionCall",*args)
	else
	  # normal calling
	  print_debug "## SendMessage: #{method}( #{args.join(", ")} )"
	  return @@driver.sendMessageToJava(method,*args)
	end
  end

  ##########################################################################
  # Public API area
  ##########################################################################

  public

  # Sending a shutdown message to the JVM.
  def break_bridge()
	return unless defined?(@@objectTable)
	@@driver.shutdownServer
  end

  # Resuming the main thread.
  def wakeup_thread()
	@@mainThread.wakeup
  end

  # Stopping the main thread so as not to finish the Ruby program.
  # (The GUI programs can wait for messages from the JVM.)
  # Calling "wakeup_thread", the Ruby program resumes the main thread.
  def stop_thread()
	return unless defined?(@@objectTable)
	@@mainThread = Thread.current
	Thread.stop
  end

  # Creating an instance of the concrete class.
  # The class name can be specified by String and Symbol.
  # Ex: "java.awt.Point" corresponds width :java_awt_Point.
  # The return value is the proxy object for the instance of the Java.
  # Corresponding notatin:  :SomeJavaClassName.jnew or :some_package_ClassName.jnew
  def jnew(classname,*args)
	__build_bridge
	__register(JCreatedObject.new(__to_s(classname),*args))
  end

  # Creating an proxy instance that can override public and protected methods of the class and interfaces.
  # Adding the singleton method to the instance, the Ruby program can implement the Java interfaces or abstract class.
  # The class name parameter can take zero or one class and arbitrary number of interfaces.
  # Those class names are specified by Symbol(only one) or String(separated by ",").
  # If Ruby class overrides the Java method that has the same, the Java method delegates the operation to Ruby method.
  # The abstract method delegates Ruby method to execute immediately.
  # Corresponding notatin:  :SomeJavaClassName.jext or :some_package_ClassName.jext
  def jextend(classnames,*args)
	__build_bridge
	__register(JExtendedClass.new(__to_s(classnames),*args))
  end

  # creating a proxy object.
  def jproxy(classname,key)
	__build_bridge
	__register(JProxyObject.new(__to_s(classname),key))
  end

  # Creating a static reference to the Java class.
  # Through the reference, the any static methods and fields can be called by Ruby.
  # Corresponding notatin:  :SomeJavaClassName.jclass or :some_package_ClassName.jclass
  def jstatic(classname)
	__build_bridge
	__register(JClass.new(__to_s(classname)))
  end

  # jimport is almost similar to "import" statement of Java.
  # Ex:
  #
  #* import "java.awt.*"
  #* import "java.awt.*,java.awt.event.*"
  #* import "java.util.List"
  #
  # The later entry is given a priority.
  def jimport(lines)
	__build_bridge
	__send_message_to_java("import",lines)
	nil
  end

  def __jclassname(key)
	__send_message_to_java("classname",key)
  end

  ##########################################################################
  # JClass area
  ##########################################################################

  private

  def __to_s(classname)
	return classname if classname.instance_of? String
	classname.to_s.gsub(/_/,".")
  end

  def __classinfo(classname)
	__send_message_to_java("classinfo",__to_s(classname)).split(",")
  end

  #
  # Abstract proxy class: a subclass should initialize @__classname, @__objectId
  #  and @__classinfo.
  #
  class JObject 

	def method_missing(name,*args)
	  print_debug "#### method_missing : #{name}( #{args.join(", ")} )"
	  name = name.to_s
	  if args.size == 0 then
		#property get?
		return __ref__(name) if @__classinfo.define_jfield?(name)
	  else
		args = __obj2id__(args)
		#property set?
		if !(name =~ /^.*=$/).nil? then
		  fname = name[0,name.length-1]
		  return __set__(fname,args[0]) if @__classinfo.define_jfield?(fname)
		end
	  end
	  #method call
	  return __id2obj__(__call__(name,*args)) if @__classinfo.define_jmethod?(name)
	  #not found
	  #super(name,*args)
	  as = args.join(",")
	  raise NoMethodError.new("Not found method: #{name}(#{as}) in #{__classname}",name, args)
	end

	# called by java
	def __jsend__(method,*args)
	  args = __id2objs__(args)
	  return __send__(method,*args)
	end

	attr_reader :__objectId,:__classname,:__classinfo

	private

	def __obj2id__(args)
	  args.map{|i|
		if @@objectTable.value?(i) then
		  i.__objectId
		elsif i.nil? then
		  NULL_SYMBOL
		else
		  i
		end
	  }
	end
	
	def __id2obj__(arg)
	  return nil if (arg.nil? || arg == NULL_SYMBOL)
	  return arg unless arg.instance_of?(String)
	  cn = __jclassname(arg)
	  if cn != NULL_SYMBOL then
		return @@objectTable[arg] if @@objectTable.key?(arg)
		jproxy(cn,arg)
	  else 
		arg
	  end
	end

	def __id2objs__(args)
	  args.map {|i| __id2obj__(i)}
	end

	def __call__(method,*args)
	  args = __obj2id__(args)
	  __id2obj__(__send_message_to_java("call",@__objectId,method,*args))
	end

	def __ref__(fieldname)
	  __id2obj__(__send_message_to_java("ref",@__objectId,fieldname))
	end

	def __set__(fieldname,value)
	  __send_message_to_java("set",@__objectId,fieldname,value)
	  nil
	end

	def to_s
	  "JB:ProxyObject  class:#{@__classname}, proxyid:#{@__objectId}"
	end

  end # class JObject

  # the object created by ruby
  class JCreatedObject < JObject
	def initialize(classname,*args)
	  if args.size > 0 then
		args = __obj2id__(args)
	  end
	  @__objectId = __send_message_to_java("new",classname,*args)
	  @__classname = __jclassname(@__objectId)
	  @__classinfo = @@class_repository.get_classinfo(@__classname)
	end
  end

  # the extended object created by ruby
  class JExtendedClass < JObject
	def initialize(_classnames,*args)
	  if args.size > 0 then
		args = __obj2id__(args)
	  end
	  @__objectId = __send_message_to_java("extend",_classnames,*args)
	  @__classname = __jclassname(@__objectId)
	  @__classinfo = @@class_repository.get_classinfo(@__classname)
	end
	
	def __jsend__(method,*args)
	  if respond_to?(method) then
		#overrided
		super(method,*args)
	  else
		raise "BUG: called not implemented method."
	  end
	end

	def __super__(method,*args)
	  args = __obj2id__(args)
	  __id2obj__(__send_message_to_java("superCall",@__objectId,method,*args))
	end

	def __call__(method,*args)
	  __super__(method,*args)
	end

	def singleton_method_added(name)
	  __send_message_to_java("impl",@__objectId,name.to_s,true)
	end

	def singleton_method_removed(name)
	  __send_message_to_java("impl",@__objectId,name.to_s,false)
	end

	def singleton_method_undefined(name)
	  __send_message_to_java("impl",@__objectId,name.to_s,false)
	end

  end

  # the object created in the JVM
  class JProxyObject <  JObject
	def initialize(classname,key)
	  @__objectId = key
	  @__classname = __jclassname(@__objectId)
	  @__classinfo = @@class_repository.get_classinfo(@__classname)
	end
  end
  
  # the static class object
  class JClass < JObject
	def initialize(classname)
	  @__objectId = __send_message_to_java("static",classname)
	  @__classname = __jclassname(@__objectId)
	  @__classinfo = @@class_repository.get_classinfo(@__classname)
	end
  end

  ##########################################################################
  # JClassRepository and JClassInfo area
  ##########################################################################

  # 
  # JClassRepository manages the class information represented by JClassInfo.
  # 
  class JClassRepository
	def initialize
	  @classtable = Hash.new
	end

	def get_classinfo(classname)
	  return nil if classname.nil?
	  info = @classtable[classname]
	  return info if info
	  info = JClassInfo.new(classname)
	  @classtable[classname] = info
	  info
	end
  end

  # 
  # JClassInfo treats superclass, public fields, public and protected methods.
  # The instances of JClassInfo make a tree form in which the subclass 
  # makes a branche.
  #
  class JClassInfo 

	attr_reader :superclass,:classname,:fields,:methods,:protected_methods,:interfaces

	def initialize(_classname)
	  info = __classinfo(_classname)
	  @classname = _classname
	  @interfaces = []
	  @fields = []
	  @methods = []
	  @protected_methods = []
	  _superclass = nil
	  mode = nil
	  info.each{|i|
		case i
		when "====Superclass"
		  mode = :superclass
		when "====Interfaces"
		  mode = :interfaces
		when "====Field"
		  mode = :fields
		when "====PublicMethod"
		  mode = :methods
		when "====ProtectedMethod"
		  mode = :protected_methods
		else
		  case mode
		  when :superclass
			_superclass = i
		  when :interfaces
			@interfaces << @@class_repository.get_classinfo(i)
		  when :fields
			@fields << i
		  when :methods
			@methods << i
		  when :protected_methods
			@protected_methods << i
		  end
		end
	  }
	  if _superclass then
		@superclass = @@class_repository.get_classinfo(_superclass) 
	  else
		@superclass = nil
	  end
	  
	end

	def define_jfield?(name)
	  return true if @fields.include? name 
	  if (!@superclass.nil?) then
		ret = @superclass.define_jfield?(name)
		return ret if ret
	  end
	  if @interfaces.size > 0 then
		ret = false
		@interfaces.each {|i|
		  if i.define_jfield?(name) then
			ret = true
			break
		  end
		}
		return true if ret
	  end
	  false
	end

	def define_jmethod?(name)
	  return true if @methods.include? name
	  return @superclass.define_jmethod?(name) if @superclass
	  false
	end

	def define_protected_jmethod?(name)
	  return true if @protected_methods.include? name
	  return @superclass.define_jmethod?(name) if @superclass
	  false
	end

	def dump
	  puts "========: #{@classname}"
	  puts "  == Superclass: #{@superclass.classname}" if @superclass
	  putter = lambda {|i| puts "   #{i}" }
	  puts "  == Interface"
	  @interfaces.each {|i| puts "   #{i.classname}"}
	  puts "  == Field"
	  @fields.each(putter)
	  puts "  == Public Method"
	  @methods.each(putter)
	  puts "  == ProtectedMethod Method"
	  @protected_methods.each(putter)
	end
  end

end #module

# Ruby like new methods
class Symbol
  def jext(*args)
	JavaBridge::jextend(self, *args)
  end
  
  def jnew(*args)
	JavaBridge::jnew(self, *args)
  end
  
  def jclass
	JavaBridge::jstatic(self)
  end
end

