package jbridge;

import java.awt.event.ActionEvent;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import javax.swing.JButton;
import jp.gr.java_conf.ccs2.core.DefaultMonitor;
import jp.gr.java_conf.ccs2.core.MessageMonitor;
import jp.gr.java_conf.ccs2.util.StringUtil;
import junit.framework.TestCase;
import javassist.*;


public class ObjectManagerTest extends TestCase {

	private DefaultMonitor mon;
	private ObjectManager manager;
	private SessionManager sessionManager;

	public void setUp() {
		mon = new DefaultMonitor(MessageMonitor.VERBOSE);
		mon.setFormat(MessageMonitor.FORMAT_MESSAGE);
		//mon.setFilters("IR.");
		mon.setDebugFlushLevel(200);

		IOverrideCall broker = new IOverrideCall() {
				public Object call(Object sid,Object objectId,String methodName,Object[] args,Class rt) {
				String[] ss = new String[args.length];
				for(int i=0;i<ss.length;i++) {
					ss[i] = args[i].toString();
				}
				mon.debug("## IR.call("+sid+": "+objectId+": "+methodName+": "+StringUtil.conbine(ss, ",")+") : "+rt.toString());
				return "SUBCLASS";
			}
		};
		IObjectTransformer transformer = new IObjectTransformer() {
				public Object exportFilter(Object arg) {
					if (arg == null) return null;
					if (arg.getClass().isPrimitive() || arg instanceof String || 
						arg instanceof Integer || arg instanceof Double || 
						arg instanceof Long || arg instanceof Short ||
						arg instanceof Byte || arg instanceof Float || 
						arg instanceof Long) {
						return arg;
					}
					return null;
				}
				public Object importFilter(Object obj) {
					return obj;
				}
			};
		sessionManager = new SessionManager(mon,broker);
		manager = new ObjectManager(mon,sessionManager,transformer);
		sessionManager.init(manager);
	}

	public void testNormalObject() throws Exception {
		//test normal object.call
		Object obj = manager.getObject(manager.createObject("jbridge.ObjectManagerTest",null));
		
		String ret = (String)manager.searchPublicMethod(obj.getClass(),"sampleMethod",null).invoke(obj,null);
		assertEquals("call:","SUPERCLASS",ret);
	}

	public void testDerivedObject() throws Exception {
		//test extend object.call -> subclass (not implemented)
		Object ext = manager.getObject(manager.extendObject("jbridge.ObjectManagerTest,java.awt.event.ActionListener",null));
		String ret = (String)manager.searchPublicMethod(ext.getClass(),"sampleMethod",null).invoke(ext,null);
		assertEquals("extend_call(not impl):","SUPERCLASS",ret);
		
		//test extend object.call -> subclass (implemented)
		Field f = ext.getClass().getField(ObjectManager.FIELD_IMPL_FLAG_PREFIX+"sampleMethod");
		f.set(ext,Boolean.TRUE);
		ret = (String)manager.searchPublicMethod(ext.getClass(),"sampleMethod",null).invoke(ext,null);
		assertEquals("extend_call(impl):","SUBCLASS",ret);

		//test extend object.call -> interface subclass 
		Object[] args = new Object[]{new ActionEvent(new JButton(),0,null)};
		ret = (String)manager.searchPublicMethod(ext.getClass(),"actionPerformed",args).invoke(ext,args);
		assertNull("interface_call:",ret);

		//test extend object.call -> super
		Object[] args2 = new Object[]{"sampleMethod",new Object[0]};
		Method mm = manager.searchPublicMethod(ext.getClass(),ObjectManager.METHOD_SUPERCLASS_METHOD,args2);
		ret = (String)mm.invoke(ext,args2);
		assertEquals("super_call:","SUPERCLASS",ret);

		//test extend object.call -> super.protected
		Object[] args1 = new Object[]{"protectedMethod",new Object[0]};
		mm = manager.searchPublicMethod(ext.getClass(),ObjectManager.METHOD_SUPERCLASS_METHOD,args1);
		ret = (String)mm.invoke(ext,args1);
		assertEquals("protected_call:","PROTECTED",ret);
	}

	public void testImport() throws Exception {
		assertNotNull(manager.staticReference("System"));

		Object id = null;
		try {
			id = manager.createObject("Point",null);
		} catch (ClassNotFoundException e) {
		}
		assertNull(id);
		manager.addImport("java.awt.*");
		id = manager.createObject("Point",null);
		Object obj = manager.getObject(id);
		assertEquals(java.awt.Point.class,obj.getClass());
	}

	public void testImportOverride() throws Exception {
		Object id = null;
		try {
			id = manager.staticReference("List");
		} catch (ClassNotFoundException e) {
		}
		assertNull(id);
		manager.addImport("java.awt.*");
		id = manager.staticReference("List");
		Object obj = manager.getObject(id);
		assertTrue(java.awt.List.class.equals(obj));

		manager.addImport("java.util.List");
		id = null;
		try {
			id = manager.staticReference("List");//can not create instance
		} catch (RuntimeException e){
		}
		obj = manager.getObject(id);
		assertTrue(java.util.List.class.equals(obj));
	}

	public void testSameHashObject() throws Exception {
		ClassPool pool = ClassPool.getDefault();
		
		CtClass ct1 = pool.makeClass("AAA");
		CtMethod m1 = CtNewMethod.make("public String call() { return \"aaa\";}",ct1);
		Object k1 = manager.obj2id(m1);
	
		CtClass ct2 = pool.makeClass("BBB");
		CtMethod m2 = CtNewMethod.make("public String call() { return \"bbb\";}",ct2);
		Object k2 = manager.obj2id(m2);

		assertTrue(m1 != m2);
		assertTrue(m1.equals(m2));
		assertFalse(k1.equals(k2));
	}

	public void testNonPublicObject() throws Exception {
		Object[] aa = {new Integer(1), new Integer(2), new Integer(3),};
		List list = Arrays.asList(aa);
		MethodFinder mf = new MethodFinder(mon);
		Method m = mf.searchPublicMethod(list.getClass(),"toArray",null);
		Object ret = m.invoke(list,null);
		assertTrue(ret instanceof Object[]);
		assertEquals(aa[0], ((Object[])ret)[0] );
	}

	//=============================

	protected String protectedMethod() {
		return "PROTECTED";
	}

	public String sampleMethod() {
		return "SUPERCLASS";
	}

	//=============================

	public void testMethodFinder() throws Exception {
		Boolean t = Boolean.TRUE, f = Boolean.FALSE;
		Object 
			ob = new Byte((byte)-1),os = new Short((short)-1),oi = new Integer(-1),
			ol = new Long(-1),of = new Float(-1.0f),od = new Double(-1);
		Object 
			ob2 = new Byte((byte)127),os2 = new Short((short)515),
			oi2 = new Integer(65539),ol2 = new Long(4294967596L);
		MethodFinder mf = new MethodFinder(mon);
		Object[] list = {
			"Byte",ob,t, "Byte",os,t, "Byte",oi,t, "Byte",ol,t, "Byte",of,f, "Byte",od,f,
			"byte",ob,t, "byte",os,t, "byte",oi,t, "byte",ol,t, "byte",of,f, "byte",od,f,

			"Short",ob,t, "Short",os,t, "Short",oi,t, "Short",ol,t, "Short",of,f, "Short",od,f,
			"short",ob,t, "short",os,t, "short",oi,t, "short",ol,t, "short",of,f, "short",od,f,

			"Integer",ob,t, "Integer",os,t, "Integer",oi,t, "Integer",ol,t, "Integer",of,f, "Integer",od,f,
			"int",ob,t, "int",os,t, "int",oi,t, "int",ol,t, "int",of,f, "int",od,f,

			"Long",ob,t, "Long",os,t, "Long",oi,t, "Long",ol,t, "Long",of,f, "Long",od,f,
			"long",ob,t, "long",os,t, "long",oi,t, "long",ol,t, "long",of,f, "long",od,f,

			"Float",ob,t, "Float",os,t, "Float",oi,t, "Float",ol,t, "Float",of,t, "Float",od,t,
			"float",ob,t, "float",os,t, "float",oi,t, "float",ol,t, "float",of,t, "float",od,t,

			"Double",ob,t, "Double",os,t, "Double",oi,t, "Double",ol,t, "Double",of,t, "Double",od,t,
			"double",ob,t, "double",os,t, "double",oi,t, "double",ol,t, "double",of,t, "double",od,t,

			"Byte",ob2,t, "Byte",os2,f, "Byte",oi2,f, "Byte",ol2,f,
			"byte",ob2,t, "byte",os2,f, "byte",oi2,f, "byte",ol2,f,

			"Short",ob2,t, "Short",os2,t, "Short",oi2,f, "Short",ol2,f,
			"short",ob2,t, "short",os2,t, "short",oi2,f, "short",ol2,f,

			"Integer",ob2,t, "Integer",os2,t, "Integer",oi2,t, "Integer",ol2,f,
			"int",ob2,t, "int",os2,t, "int",oi2,t, "int",ol2,f,

			"Long",ob2,t, "Long",os2,t, "Long",oi2,t, "Long",ol2,t,
			"long",ob2,t, "long",os2,t, "long",oi2,t, "long",ol2,t,

			"Float",ob2,t, "Float",os2,t, "Float",oi2,t, "Float",ol2,t, 
			"float",ob2,t, "float",os2,t, "float",oi2,t, "float",ol2,t, 

			"Double",ob2,t, "Double",os2,t, "Double",oi2,t, "Double",ol2,t,
			"double",ob2,t, "double",os2,t, "double",oi2,t, "double",ol2,t,

			"Boolean",Boolean.TRUE,t, "Boolean",Boolean.FALSE,t,
			"boolean",Boolean.TRUE,t, "boolean",Boolean.FALSE,t,
		};

		int counter = 0;
		while (true) {
			String name = (String)list[counter++];
			Object arg = list[counter++];
			boolean ss = (list[counter++] == Boolean.TRUE);
			MethodFinderTester test = new MethodFinderTester(name,arg,ss);
			assertTrue("MD Test: name="+name+" ("+arg.getClass().getName()+") : "+ss,
					   test.execTest(mf));
			if (counter == list.length) break;
		}
	}

	class MethodFinderTester {
		private String name;
		private Object arg;
		private boolean ss;
		MethodFinderTester(String n,Object a,boolean s) {
			name = n; arg = a; ss = s;
		}
		boolean execTest(MethodFinder mf) throws Exception {
			boolean test = false;
			try {
				Method m = mf.searchPublicMethod(ObjectManagerTest.class,"mf_"+name,new Object[]{arg});
				test = (m != null);
			} catch (NoSuchMethodException e) {
				//e.printStackTrace();
				test = false;
			}
			return test == ss;
		}
	}

	public boolean mf_boolean(boolean a) { return a; }
	public Boolean mf_Boolean(Boolean a) { return a; }

	public byte mf_byte(byte a) { return a; }
	public short mf_short(short a) { return a; }
	public int mf_int(int a) { return a; }
	public long mf_long(long a) { return a; }
	public float mf_float(float a) { return a; }
	public double mf_double(double a) { return a; }

	public Byte mf_Byte(Byte a) { return a; }
	public Short mf_Short(Short a) { return a; }
	public Integer mf_Integer(Integer a) { return a; }
	public Long mf_Long(Long a) { return a; }
	public Float mf_Float(Float a) { return a; }
	public Double mf_Double(Double a) { return a; }

}