/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.sql;

import java.sql.*;
import jp.gr.java_conf.ccs2.util.StringUtil;
import java.util.Map;
import java.util.HashMap;
import java.util.ResourceBundle;
import java.util.MissingResourceException;

/**
   <code>JDBCUtil</code> has some utilities to connect and handle 
   a DB source. 
  
   [for Java2]

   @author SAKURAI Masashi
 */

public class JDBCUtil {

	/**
	   Store column data of a row into a HashMap object.
	   The result set should on the target position.
	   The given hash map will be cleared.
	   The column name is transformed into lower case.
	 */
	public static void row2map(ResultSet set,HashMap map) throws SQLException {
		map.clear();
		ResultSetMetaData rsmd = set.getMetaData();
		int cols = rsmd.getColumnCount();
		for (int i=0;i<cols;i++) {
			String colName = rsmd.getColumnName(i+1).toLowerCase();
			map.put(colName,set.getObject(i+1));
		}
	}
	
	//===================================
	// DB Connection
	//===================================

	private static Map driverMap = new HashMap();
	
	public static boolean registDriver(String driverClassName) {
		if (driverMap.get(driverClassName) != null) return true;
		try {
			driverMap.put(driverClassName,
						  Class.forName(driverClassName).newInstance());
			return true;
		}
		catch (Exception e) {
			System.err.println("Unable to load driver ["+driverClassName+"].");
			e.printStackTrace();
		}
		return false;
	}

	//===================================
	// Native DB Command
	//===================================

	private static ResourceBundle dbResource;
	public static NativeDBCommand getNativeDBCommand(Statement stat) throws SQLException {
		return getNativeDBCommand(stat.getConnection());
	}

	public static NativeDBCommand getNativeDBCommand(Connection con) throws SQLException {
		String name	 = con.getMetaData().getDatabaseProductName();
		name = StringUtil.replace(name," ","_");
	
		if (dbResource == null) {
			dbResource = ResourceBundle.getBundle("jp.gr.java_conf.ccs2.sql.nativedb");
		}
		String cname = "";
		try {
			cname = dbResource.getString(name);
			Class cls = Class.forName(cname);
			return (NativeDBCommand)cls.newInstance();
		} catch (MissingResourceException e) {
			System.err.println("Missing DB class name ["+name+"]");
			e.printStackTrace();
		}  catch (InstantiationException e) {
			System.err.println("InstantiationException ["+
							   e.getMessage()+"]");
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			System.err.println("IllegalAccessException ["+
							   e.getMessage()+"]");
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			System.err.println("can not load the class ["+cname+"]");
			e.printStackTrace();
		}
		throw new InternalError();
	}

	/**
	   calculate row number with rs.last() and rs.getRow().
	   some database drivers will fail in this method, because 
	   they can not treat rs.beforeFirst()
	   @param rs result set object
	   @return row number
	 */
	public static int getRowNumber(ResultSet rs) throws SQLException {
		if (rs == null) return 0;
		rs.last();
		int cc = rs.getRow();
		rs.beforeFirst();
		return cc;
	}

	private static HashMap tableMap = new HashMap();

	/**
	   @param con 
	   @param database 
	   @param table
	   @param column
	   @return type number described in java.sql.Types.
	 */
	public static int getColumnType(Connection con,String table,
									String column) throws SQLException {
		HashMap columnOfTable = (HashMap)tableMap.get(table);
		if (columnOfTable == null) {
			columnOfTable = new HashMap();
			tableMap.put(table,columnOfTable);
		}
		Integer it = (Integer)columnOfTable.get(column);
		if (it != null ) {
			return it.intValue();
		}
		DatabaseMetaData meta = con.getMetaData();
		ResultSet rs = meta.getColumns(con.getCatalog(),"",table,column);
		if (rs.next()) {
			int tp = rs.getInt("DATA_TYPE");
			columnOfTable.put(column,new Integer(tp));
			return tp;
		}
		throw new SQLException("No such column ... ["+table+"."+column+"]");
	}

	public static boolean isCharTypeColumn(Connection con,
										   String table,String column) throws SQLException {
		int type = getColumnType(con,table,column);
		if (type == Types.VARCHAR ||
			type == Types.CHAR ||
			type == Types.TIME ||
			type == Types.TIMESTAMP ||
			type == Types.DATE ||
			type == Types.LONGVARCHAR) {
			return true;
		}
		return false;
	}

	public static String makeSqlConstant(Statement stat,String table,String column,String in) throws SQLException {
		return makeSqlConstant(stat.getConnection(),table,column,in);
	}

	public static String makeSqlConstant(Connection con,String table,String column,String in) throws SQLException {
		if (isCharTypeColumn(con,table,column)) {
			return "'"+escape(in,con)+"'";
		}
		return in;
	}

	public static String escape(String in,Statement stat) {
		try {
			return escape(in,stat.getConnection());
		} catch (SQLException e) {
			System.err.println("SQLException : "+e.getMessage());
		}
		return defaultEscape(in);
	}

	public static String escape(String in,Connection con) {
		try {
			NativeDBCommand com = getNativeDBCommand(con);
			return com.escape(in);
		} catch (SQLException e) {
			System.err.println("SQLException : "+e.getMessage());
		}
		return defaultEscape(in);
	}

	private static char [] orgChars = {'\\','\'','%'};
	private static String [] replaceChars = {"\\\\","\\'","\\%"};

	public static String defaultEscape(String text) {
		return StringUtil.escape(text,orgChars,replaceChars);
	}
}


