/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.util;
import java.io.*;
import jp.gr.java_conf.ccs2.util.StringUtil;
import java.text.SimpleDateFormat;
import jp.gr.java_conf.ccs2.io.FileSystemUtil;
import java.util.GregorianCalendar;
import java.util.Calendar;
import java.util.Date;


/**
 * This class back up a target file into the "yyyy/mm/dd" format directory.
 * Then, this class makes internal information file into the log directory.
 */
public class DateFileRotator implements FileRotator {
	
	private File baseDir;
	private File targetFile;

	private String infoFile = ".ccs_lastRatatedTime";

	private long lastBackupedTime = -1;

	private int backupIntervalDate = 7;//days
	
	private static SimpleDateFormat DF = new SimpleDateFormat("yyyy/MM/dd");

	public DateFileRotator(String targetFile,int backupIntervalDate) {
		setFilename(targetFile);
		this.backupIntervalDate = backupIntervalDate;
	}

	public void setFilename(String path) {
		this.targetFile = new File(path).getAbsoluteFile();
		baseDir = targetFile.getParentFile();
		lastBackupedTime = -1;
	}

	public String getFilename() {
		return targetFile.getPath();
	}

	public void rotate() {
		if (!doesRotate()) return;
		backup();
	}

	/**
	 * create File object from baseDir and infoFile.
	 */
	private File getInfoFile() {
		return new File(baseDir.getPath()+"/"+infoFile);
	}

	private boolean doesRotate() {
		if (lastBackupedTime < 0) {
			File info = getInfoFile();
			if (!info.exists()) return true;
			String stime = FileSystemUtil.file2String(info.getPath());
			if (!StringUtil.isNull(stime)) {
				try {
					lastBackupedTime = getTheDayFirst(Long.parseLong(stime));
				} catch (NumberFormatException e) {
				}
			} else {
				return true;
			}
		}
		return System.currentTimeMillis() > getNextBackupTime();
	}

	/**
	 * calculate next backup time from lastBackupedTime and backupIntervalDate.
	 */
	private long getNextBackupTime() {
		return lastBackupedTime + 24*60*60*1000*backupIntervalDate;
	}

	private static long getTheDayFirst(long day) {
		GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(new Date(day));
		GregorianCalendar first = 
			new GregorianCalendar(cal.get(Calendar.YEAR),
								  cal.get(Calendar.MONTH),
								  cal.get(Calendar.DATE));
		return first.getTime().getTime();
	}

	private void backup() {
		File backupDir = new File(baseDir,DF.format(new Date()));
		if (!backupDir.exists()) {
			backupDir.mkdirs();
		}
		File backupFile = new File(backupDir,targetFile.getName());
		targetFile.renameTo(backupFile);
		lastBackupedTime = getTheDayFirst(System.currentTimeMillis());
		FileSystemUtil.string2File(Long.toString(lastBackupedTime),getInfoFile().getPath());
	}

	public static void main(String [] args) throws IOException {
		String file = "test.log";
		new File(file).createNewFile();
		DateFileRotator dfr = new DateFileRotator(file,1);
		dfr.rotate();
	}
}
