package jbridge;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import jp.gr.java_conf.ccs2.core.DefaultMonitor;
import jp.gr.java_conf.ccs2.core.MessageMonitor;


public class InvocationPool {

	private int maxJob   = 4;
	private LinkedList jobQueue = new LinkedList();
	private Worker[] workers;
	private MessageMonitor monitor;

	public InvocationPool(MessageMonitor mon,int maxWorker,int maxJob) {
		this.maxJob = maxJob;
		this.monitor = mon;
		workers = new Worker[maxWorker];
		for(int i=0;i<maxWorker;i++) {
			workers[i] = new Worker();
		}
	}

	public void invokes(Runnable job) {
		monitor.debug("IPool:  Job coming : "+job.toString());
		while(true) {
			synchronized(jobQueue) {
				if (jobQueue.size() < maxJob) {
					jobQueue.add(job);
					jobQueue.notifyAll();
					break;
				}
				try {
					monitor.debug("IPool:  waiting for queue... ");
					jobQueue.wait();
				} catch (InterruptedException e) {
					monitor.recordStackTrace(e);
				}
			}
		}
	}

	public void dispose() {
		for(int i=0;i<workers.length;i++) {
			workers[i].finish();
		}
		synchronized(jobQueue) {
			jobQueue.notifyAll();
		}
	}

	private class Worker {

		private boolean[] finishFlag = {false};

		private Thread thread = new Thread(new Runnable() {
				public void run() {
					try {
						workloop();
					} catch (Throwable t) {
						monitor.recordStackTrace(t);
					} finally {
						monitor.verbose("InvocationPool: thread finished.");
					}
				}
			});

		Worker() {
			thread.start();
		}

		void finish() {
			synchronized(finishFlag) {
				finishFlag[0] = true;
			}
		}

		private void workloop() {
			while(true) {
				Runnable job = null;
				synchronized(jobQueue) {
					if (jobQueue.size() > 0) {
						job = (Runnable)jobQueue.removeFirst();
						jobQueue.notifyAll();
					} else {
						try {
							jobQueue.wait();
						} catch (InterruptedException e) {
							monitor.recordStackTrace(e);
						}
					}
				}
				if (job != null) {
					job.run();
				}
				synchronized(finishFlag) {
					if (finishFlag[0]) {
						break;
					}
				}
			}
		}
		
	}

	//=================================================================
	// BENCH MARK TEST
	//=================================================================

	public static void main(String[] args) {
		int num = 1000;
		DefaultMonitor mon = new DefaultMonitor();
		long threadTime = 0,poolTime = 0,single = 0;
		int maxjob = 100;
		int maxworker = 10;
		InvocationPool pool = new InvocationPool(mon,maxworker,maxjob);
		for(int i=0;i<3;i++) {
			System.out.println("Thread");
			threadTime += testThread(num,mon);
			System.out.println(""+threadTime);
			System.out.println("Pool");
			poolTime += testPool(num,pool,mon);
			System.out.println(""+poolTime);
			System.out.println("Single");
			single += testSingle(num,mon);
			System.out.println(""+single);
			System.out.println("##");
		}
		System.out.println("-----------------------");
		System.out.println("Thread test");
		System.out.println(""+threadTime);
		System.out.println("Pool   test");
		System.out.println(""+poolTime);
		System.out.println("Single test");
		System.out.println(""+single);
		pool.dispose();
	}

	private static long testPool(int num,InvocationPool pool,MessageMonitor mon) {
		LinkedList results = new LinkedList();
		Runnable[] jobs = makeJobs(results,num,mon);
		long start = System.currentTimeMillis();
		for(int i=0;i<jobs.length;i++) {
			pool.invokes(jobs[i]);
		}
		wait(results,num,mon);
		return System.currentTimeMillis() - start;
	}

	private static long testThread(int num,MessageMonitor mon) {
		LinkedList results = new LinkedList();
		Runnable[] jobs = makeJobs(results,num,mon);
		long start = System.currentTimeMillis();
		for(int i=0;i<jobs.length;i++) {
			new Thread(jobs[i]).start();
		}
		wait(results,num,mon);
		return System.currentTimeMillis() - start;
	}

	private static long testSingle(int num,MessageMonitor mon) {
		LinkedList results = new LinkedList();
		Runnable[] jobs = makeJobs(results,num,mon);
		long start = System.currentTimeMillis();
		for(int i=0;i<jobs.length;i++) {
			jobs[i].run();
		}
		wait(results,num,mon);
		return System.currentTimeMillis() - start;
	}

	private static void wait(List results,int num,MessageMonitor mon) {
		while(true) {
			synchronized(results) {
				if (results.size() == num) {
					break;
				}
				try {
					results.wait();
				} catch (InterruptedException e) {
					mon.recordStackTrace(e);
				}
			}
		}
	}

	private static Runnable[] makeJobs(List rs,int jobs,MessageMonitor m) {
		final List results = rs;
		final MessageMonitor mon = m;
		Runnable[] ret = new Runnable[jobs];
		final int wait = 100000;
		for(int i=0;i<jobs;i++) {
			final int aaa = i;
			ret[i] = new Runnable() {
					public void run() {
						int timer = (int)(Math.random()*100);
						for(int i=0;i<timer;i++) {
							Date d = new Date(System.currentTimeMillis()-(long)(Math.random()*1000000));
							String a = d.toString();
							a.length();
						}
						synchronized(results) {
							results.add(new Integer(aaa));
							results.notify();
						}
					}
				};
		}
		return ret;
	}

}
