package jbridge.comm.binstream;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import jp.gr.java_conf.ccs2.core.DefaultMonitor;
import jp.gr.java_conf.ccs2.core.MessageMonitor;


public class BinServer implements BinConstants,ICommunicator {

	private MessageMonitor monitor;
	private ServerSocket serverSocket;
	private ServiceManager handlerManager = new ServiceManager();
	private MessageServer messageServer;

	private Thread socketThread;
	private boolean shutdownFlag = false;

	public BinServer(MessageMonitor mon,int port) throws IOException {
		serverSocket = new ServerSocket(port);
		this.monitor = mon;
		messageServer = new MessageServer("SV",monitor,handlerManager);
	}

	public BinServer(MessageMonitor mon) throws IOException {
		this(mon,0);
	}

	public int getPortNumber() {
		return serverSocket.getLocalPort();
	}

	public void start() {
		socketThread = new Thread(acceptor);
		socketThread.setName("BSV|socket");
		socketThread.start();
	}

	public void shutdown() {
		monitor.debug("BinServer: shutdown signal arrived.");
		shutdownFlag = true;
		messageServer.shutdown();
		while(true) {
			if (socketThread == null) {
				break;
			}
			try {
				Thread.sleep(300);
			} catch (InterruptedException e) {
				monitor.recordStackTrace(e);
				return;
			}
		}
		monitor.debug("BinServer: shutdown completed.");
	}

	private Runnable acceptor = new Runnable() {
			public void run() {
				try {
					while(!shutdownFlag) {
						Socket s = null;
						try {
							monitor.verbose("BinServer: waiting for client's connection...");
							messageServer.setSocket(waitingForClientConnection(serverSocket));
							monitor.verbose("BinServer: connection established.");
							messageServer.blockWorkingThread();	
							monitor.verbose("BinServer: disconnected.");
						} catch (SocketTimeoutException e) {
							continue;
						} catch (IOException e) {
							monitor.recordStackTrace(e);
							return;
						}
					}
				} finally {
					monitor.debug("BinServer: Acceptor thread finished.");
					socketThread = null;
				}
			}
		};
	
	private Socket waitingForClientConnection(ServerSocket ss) throws IOException {
		ss.setSoTimeout(300);
		while(true) {
			try {
				return ss.accept();
			} catch (SocketTimeoutException e) {
				if (shutdownFlag) {
					throw e;
				}
				continue;
			}
		}
	}

	public void addHandler(String name,IMessageHandler h) {
		handlerManager.addHandler(name,h);
	}

	public Object send(String name,Object[] args) throws IOException {
		return messageServer.send(name,args);
	}

	public void removeHandler(String name) {
		handlerManager.removeHandler(name);
	}

	//===== echo server

	public static void main(String[] args) throws Exception {
		int level = MessageMonitor.VERBOSE;
		if (args.length > 1) {
			if ("debug".equals(args[1])) {
				level = MessageMonitor.DEBUG;
			} else if ("verbose".equals(args[1])) {
				level = MessageMonitor.VERBOSE;
			} else if ("normal".equals(args[1])) {
				level = MessageMonitor.NORMAL;
			}
			System.out.println("LogLevel:"+level+"  <- "+MessageMonitor.VERBOSE);
		}
		DefaultMonitor mon = new DefaultMonitor(level);
		mon.setFormat(MessageMonitor.FORMAT_MESSAGE);
		int port = Integer.parseInt(args[0]);
		BinServer server = new BinServer(mon,port);
		server.addHandler("echo_int1",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Byte((byte)(((Number)arg[0]).byteValue()*2));
				}
			});
		server.addHandler("echo_int2",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Short((short)(((Number)arg[0]).shortValue()*2));
				}
			});
		server.addHandler("echo_int4",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Integer(((Number)arg[0]).intValue()*2);
				}
			});
		server.addHandler("echo_int8",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Long(((Number)arg[0]).longValue()*2);
				}
			});
		server.addHandler("echo_str",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					String s = (String)arg[0];
					return s+s;
				}
			});
		server.addHandler("echo_decimal",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					BigDecimal s = null;
					if (arg[0] instanceof BigDecimal) {
						s = (BigDecimal)arg[0];
					} else {
						s = new BigDecimal(arg[0].toString());
					}
					return s.multiply(new BigDecimal("2"));
				}
			});
		server.addHandler("echo_float",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Float(((Number)arg[0]).floatValue()*2.0f);
				}
			});
		server.addHandler("echo_double",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					return new Double(((Number)arg[0]).doubleValue()*2.0);
				}
			});
		server.addHandler("echo_int1_array",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					byte[] ba = (byte[])arg[0];
					for(int i=0;i<ba.length;i++) {
						ba[i] *= 2;
					}
					return ba;
				}
			});
		server.addHandler("echo_int4_array",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					int[] ba = (int[])arg[0];
					for(int i=0;i<ba.length;i++) {
						ba[i] *= 2;
					}
					return ba;
				}
			});
		server.addHandler("echo_str_array",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					String[] ba = (String[])arg[0];
					for(int i=0;i<ba.length;i++) {
						ba[i] = ba[i]+ba[i];
					}
					return ba;
				}
			});
		final boolean[] stopper = {false};
		server.addHandler("exit",new IMessageHandler() {
				public Object send(Object[] arg) throws Exception {
					synchronized(stopper) {
						stopper[0] = true;
						stopper.notifyAll();
					}
					return null;
				}
			});
		server.start();
		System.out.println("Server started on port["+port+"]. Send [exit] message to shutdown this server.");
		while(true) {
			synchronized(stopper) {
				if (stopper[0]) {
					Thread.sleep(300);
					server.shutdown();
					Thread.sleep(300);
					System.exit(0);
				}
				stopper.wait();
			}
		}
	}
}
