/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.comp;
import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;


/**
 * <p>This class make an effected title panel.</p>
 * usage:<br>
 <code>
 Shader sh = new Shader("title",  // title you want to show
 0xcccccc,// shade color
 0xffffff,// back ground color
 0x99bb99,// title color
 "Serif",// font name
 35,// font size
 Shader.ITALIC); // font style
 some_component.add(sh);
 </code>
*/

public class Shader extends Canvas implements Runnable {

	//==========================
	// field properties
	//==========================

	//public meber

	/** font style */
	public static final int PLAIN = 0;
	/** font style */
	public static final int ITALIC = 1;
	/** font style */
	public static final int BOLD = 2;

	//default

	public static final int	   defaultRight = 7;
	public static final int	   defaultDown = 6;
	public static final int	   defaultForColor = 0xcccccc;
	public static final int	   defaultBackColor = 0xffffff;
	public static final int	   defaultTitleColor = 0x99bb99;
	public static final int	   defaultDepth = 10;
	public static final String defaultFontname = "Serif";
	public static final int	   defaultFontstyle = 0;
	public static final int	   defaultFontsize = 30;


	//base parameter
	String		title;		 //title string
	int		backColor,forColor,titleColor;//color
	int		depth;		 //shade depth
	int			right,down;	 //shade offset
	int			wright,wdown;//shade way
	int		width,height;//component size
	String		fontname;	 //font name
	int			fontstyle;	 //font style
	int			fontsize;	 //font size

	//parent component
	MessageListener parent = null;

	//special and temporary parameter

	double		frexx = 0.;
	double		frexy = 0.;
	int			frontdepth = 2;

	int			xoffset,yoffset;
	int		lastWidth,lastHeight;//last component size
	boolean		fix = true;
	boolean		selfResize = false;
	
	//	@@ Graphics

	Image	ibackOffice;
	Graphics	gbackOffice;
	Image		imgr;

	//	@@ Thread

	int		isDraw;
	boolean paintRequest = false;
	Thread	writer;
	static final int	 READY=0,PAINTING=1,FINISHED=2,FIRST=3;

	//==========================
	// test method 
	//==========================

	/** stand alone method.(example of usage) */
	public static void main(String arg[]) {
		String put="Shader sample!!";
		int fg = 0xcccccc;
		int bg = 0xffffff;
		int wc = 0x99bb99;
		if (arg.length!=4) {
			System.out.println("stand alone usage sample:");
			System.out.println("% java jp.gr.java_conf.ccs2.comp.Shader [word] [0x000000:fg color] [0x000000:bg color] [0x000000:word color]");
		} else {
			put = arg[0];
			fg = Integer.decode(arg[1]).intValue();
			bg = Integer.decode(arg[2]).intValue();
			wc = Integer.decode(arg[3]).intValue();
		}
		JFrame f = new JFrame("Shader example");
		f.addWindowListener(new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					System.exit(0);
				}
			});
		Shader sh = new Shader(put,
							   fg,bg,wc,
							   15,0,0,
							   "Serif",40,Shader.ITALIC);
		f.getContentPane().add(sh,"Center");
		f.pack();
		f.show();
	}

	//==========================
	// constructors
	//==========================

	/** no parameter (depend on title size) */
	public Shader() { this("Shader Component"); }

	/** only title	(depend on title size) */
	public Shader(String title) {
		this(title,defaultForColor,defaultBackColor,defaultTitleColor,
			 defaultFontname,defaultFontsize,defaultFontstyle);
	}

	/** title, colors and font style  (depend on title size) */
	public Shader(String title,int fcol,int bcol,int wcol,
				  String font,int large,int style) {
		this(title,fcol,bcol,wcol,
			 defaultDepth,defaultRight,defaultDown,
			 font,large,style);
	}

	/** title, colors, shade parameters and font style
	 * (depend on title size) */
	public Shader(String title,int fcol,int bcol,int wcol,
				  int depth,int right,int down,
				  String font,int large,int style) {
		this(title,fcol,bcol,wcol,
			 depth,right,down,
			 font,large,style,
			 0,0);
		selfResize = true;
	}

	/** title and size of component
	 * (depend on component size)
	 */
	public Shader(String title,int width,int height) {

		this(title,defaultForColor,defaultBackColor,defaultTitleColor,
			 defaultDepth,defaultRight,defaultDown,
			 defaultFontname,0,defaultFontstyle,
			 width,height);
	}

	/** title, colors,shade parameters, font style and size of component
	 * (depend on component size)
	 */
	public Shader(String title,int fcol,int bcol,int wcol,
				  int depth,int right,int down,
				  String font,int style,
				  int width,int height) {

		this(title,fcol,bcol,wcol,
			 depth,right,down,
			 font,0,style,
			 width,height);
	}

	/** all parameter and image */
	public Shader(String title,int fcol,int bcol,int wcol,
				  int depth,int right,int down,
				  String font,int large,int style,
				  int width,int height) {
		//triangle function
		for (int i=0;i<320;i++) {
			data[i]=(int)(Math.sin((i%256)*Math.PI/128)*1024);
		}
	
		this.title = title;
		this.width = width;
		this.height = height;
		this.depth=depth;
		this.right=right;
		this.down=down;
		if (this.right<0) {
			wright=-this.right;
			this.right=0;
		} else wright=0;
		if (this.down<0) {
			wdown=-this.down;
			this.down=0;
		} else wdown=0;

	
		titleColor=wcol;
		backColor=bcol;
		forColor=fcol;

		this.depth = depth;
		fontname = font;
		fontsize = large;
		fontstyle = style;
	
		if (width != 0) {
			setSize(width,height);
		} else {
			this.width = fontsize*title.length();
			this.height = fontsize*2;
			setSize(this.width,this.height);
		}

		setDraw(FIRST);
	}

	//==========================
	// operation
	//==========================

	/** set size of this component. Even if font size was given,
	 * this component resizes and adjusts font size.
	 */
	public void setSize(int w,int h) {
		selfResize = false;
		super.setSize(w,h);
	}

	public Dimension getMinimumSize() {
		return new Dimension(width,height);
	}
	public Dimension getPreferredSize() {
		Dimension d = new Dimension(width,height);
		return d;
	}

	/** set size of this component. Even if font size was given,
	 * this component resizes and adjusts font size.
	 */
	public void setSize(Dimension d) {
		selfResize = false;
		super.setSize(d);
	}

	/** change title */
	public void setTitle(String title) {
		this.title = title;
		urgeRepaint();
	}

	public String getTitle() {return title;}

	//==========================
	// over ride 
	//==========================

	public void paint(Graphics g) {
		branch(g);
		if (paintRequest) {
			paintRequest = false;

			Runnable updateAComponent = new Runnable() {
					public void run() { repaint(); }
				};
			SwingUtilities.invokeLater(updateAComponent);
		}
	}


	public void run()  {
		lastWidth = width;
		lastHeight = height;
		ibackOffice=this.createImage(width,height);
		gbackOffice=ibackOffice.getGraphics();
		makeGraph(gbackOffice);
		setDraw(FINISHED);

		if (parent != null) parent.message("",0,new Dimension(width,height));

		invalidate();

		Runnable updateAComponent = new Runnable() {
				public void run() { repaint(); }
			};
		SwingUtilities.invokeLater(updateAComponent);
	}

	public void addNotify() {
		super.addNotify();
		if (selfResize) {
			FontMetrics fm = getFontMetrics(new Font(fontname,
													 Font.PLAIN,(int)((fontsize+depth)*3)));
			xoffset = (Math.abs(right)+Math.abs(wright))*2;//depth/2;
			yoffset = (Math.abs(down)+Math.abs(wdown))*2;//depth/2;
			setSize(fm.stringWidth(title)+xoffset,
					fm.getHeight()+yoffset);
		}
	}
   
	//==========================
	// private area
	//==========================

	synchronized private void setDraw(int a) {isDraw = a;}
	synchronized private int getDraw() {return isDraw;}
	
	private void urgeRepaint() {
		if (getDraw() == FINISHED) 
			setDraw(READY);
		Runnable updateAComponent = new Runnable() {
				public void run() { repaint(); }
			};
		SwingUtilities.invokeLater(updateAComponent);
	}

	synchronized private void branch(Graphics g) {
		Dimension d = getSize();
		width = d.width;
		height = d.height;

		switch (getDraw()) {
		default:
		case FIRST:
		case READY:
			setDraw(PAINTING);
			threadStart();
			break;
		case PAINTING:
			/*
			  g.setColor(new Color(backColor));
			  g.fillRect(0,0,width,height);
			  g.setColor(new Color(titleColor));
			  g.drawString(title+" (now painting...)",10,20);
			*/
			break;
		case FINISHED:
			//check the resize
			if (ibackOffice == null ||
				width != lastWidth	|| height != lastHeight) {
				setDraw(READY);
				paintRequest = true;
				break;
			} else 
				g.drawImage(ibackOffice,0,0,this);
			writer = null;
		}
	}


	private String [] splitString(String t) {
		t = t.replace('~',' ');
		int length = t.length();
		String [] ws = new String [length];
		for (int i = 0;i<length;i++) {
			ws[i] = new String(t.substring(i,i+1));
		}
		return ws;
	}

	private Font [] makeFonts(String name,int style,
							  int baseSize,int d) {
		int tstyle = Font.PLAIN;
		if ((style & PLAIN) != 0) {
			tstyle = Font.BOLD;
		}
		if ((style & ITALIC) != 0) {
			tstyle += Font.ITALIC;
		}
		Font [] fs = new Font[d];
		for (int i=0; i<d; i++) {
			fs[i] = new Font(name,tstyle,baseSize+i);
		}
		return fs;
	}

	private Color [] makeColors(int d,int fcol,int bcol,int wcol) {
		int br,bg,bb,cr,cg,cb;
		double fr,fg,fb;
		br = (backColor>>16)%256;
		bg = (backColor>>8)%256;
		bb = backColor%256;
		fr = (forColor>>16)%256;
		fg = (forColor>>8)%256;
		fb = (forColor%256);
		Color [] colorIndex = new Color [d];
		for (int i=0; i<d; i++) {
			cr = br + ((int)((fr-br)*i*i/d/d))%256;
			cg = bg + ((int)((fg-bg)*i*i/d/d))%256;
			cb = bb + ((int)((fb-bb)*i*i/d/d))%256;
			if (cb>255) cb = 255;
			if (cg>255) cg = 255;
			if (cr>255) cr = 255;
			colorIndex[(depth-i)-1] = new Color(cr,cg,cb);
		}
		return colorIndex;
	}
	
	private void threadStart() {
		if (writer == null) {
			writer = new Thread(this);
			writer.start();
		}
	}

	private synchronized void makeGraph(Graphics g) {
		String [] words = splitString(title);
		int length = title.length();
		Color [] colorIndex = makeColors(depth,forColor,backColor,titleColor);
		xoffset = Math.abs(right)+Math.abs(wright);//depth/2;
		yoffset = Math.abs(down)+Math.abs(wdown);//depth/2;

		FontMetrics fm;
		{
			Font tf;
			int tx,ty;
			double mp;
			if (fontsize == 0 || fix) {
				fix = true;
				tx = width/length;
				ty = height/2;
				fontsize = (tx>ty) ? ty : tx;
			}
			while(true) {
				tf = new Font(fontname,fontstyle,fontsize+depth-2);
				g.setFont(tf);
				fm = g.getFontMetrics();
				tx = 2*xoffset;
				for (int i=0; i<length;i++) {
					tx += fm.stringWidth(words[i]);
				}
				ty = fm.getHeight()+2*yoffset;
				if (tx < width && ty < height) {
					xoffset = width/2-tx/2;
					break;
				}
				if (tx < width)
					mp = (double)width/tx;
				else
					mp = (double)height/ty;
				if (mp < 0.8) 
					fontsize *= mp;
				fontsize-=3;
				if (fontsize < 9) {
					fontsize = 9;
					break;
				}
			}
		}
		Font [] fonts = makeFonts(fontname,fontstyle,fontsize,depth);

		//
		int tx,tt,ph,dx,dy,ii;
		int checkx,checky,cx,cy;
		//variable list
		//tx:[vł̊exW
		//tt:~
		//ph:]p (360/tt)
		//ii:]a
		//dx,dy:]ʒu
		//checkx,chexky
		g.setColor(colorIndex[depth-1]);
		g.fillRect(0,0,width,height);
		g.setFont(fonts[depth-1]);
		fm = g.getFontMetrics();
		int [] ddx = new int [length];
		for (int i=0; i<length;i++) {
			ddx[i] = fm.stringWidth(words[i]);
		}
		yoffset += height/2+(fm.getAscent()-fm.getDescent())/4;
		checkx=0;
		checky=0;
		cx=0;cy=0;
	
		//O[vFe̊eZ
		for (int i=depth-1; i>0; i--) {
			ii = i-1;
			g.setColor(colorIndex[i]);
			g.setFont(fonts[i]);
			tt = i*frontdepth;
			if (i == (depth-1)) ii += 2;
			if (i == (depth-2)) ii += 1;
			for (int jj=0; jj<tt; jj++) {
				//[vF̉]
				ph = (int)(256*jj/tt);
				dx = i+4+(ii*cos(ph))>>10;
				dy = i+(ii*sin(ph))>>10;
				tx = 0;
				//e\
				cx = xoffset+right+dx;
				cy = yoffset+down+dy;
				for (int j=0;j<length;j++) {
					g.drawString(words[j],cx+tx,cy);
					tx += ddx[j]-2;
				}
				cx += tx;
				//Kv̈vZp
				if (checkx<cx) checkx=cx;
				if (checky<cy) checky=cy;
			}
		}

		g.setColor(new Color(titleColor));
		tx = 0;
		for (int j=0;j<length;j++) {
			g.drawString(words[j],
						 xoffset+tx+wright+(int)(frexx*depth),
						 yoffset+wdown+(int)(frexy*depth));
			tx += ddx[j]-2;
		}

		//Kv̈
		/*
		  checkx+=fm.stringWidth(words[length-1])+wright;
		  checky+=fm.getDescent()+wdown;
		  width = checkx;
		  height = checky;
		  setSize(checkx,checky);
		  //System.out.println(title+" = "+checkx+" "+checky);
		  */
	}
	
	int data [] = new int [320];

	int sin(int deg) {
		deg = deg & 255;
		return data[deg];
	}
	
	int cos(int deg) {
		deg = (deg & 255) + 64;
		return data[deg];
	}
}
