/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.comp;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JFrame;
import javax.swing.JScrollPane;

import jp.gr.java_conf.ccs2.util.TreeStructure;
import jp.gr.java_conf.ccs2.util.TreeStructureUtil;
import javax.swing.JComponent;


/** TreeStructure data viewer implemented by Graphics tree */
public class TreeStructureViewer {

	public TreeStructureViewer(TreeStructure root) {
		setTree(root);
	}

	public void setTree(TreeStructure root) {
		show(root);
	}

	private JFrame frame;
	private TreeViewer tree;

	protected void show(TreeStructure root) {
		if (frame == null) {
			frame = new JFrame("Tree Viewer");
			frame.setSize(600,500);
		}
		//
		if (tree == null) {
			tree = new TreeViewer(root);
			JScrollPane sc = new JScrollPane(tree);
			frame.getContentPane().add(sc);
		} else {
			tree.setModel(root);
		}
		frame.show();
	}
	
	class TreeViewer extends JComponent {

		int vSpace = 15;
		int hSpace = 15;

		OriginalFont fontInfo = 
			new OriginalFont(Color.black,Color.lightGray,
							 new Font("Serif",Font.PLAIN,16));

		TreeViewer(TreeStructure t) {
			setModel(t);
		}

		public void setModel(TreeStructure t) {
			root = t;
		}

		TreeStructure root;
		List paintList = new ArrayList();

		public void paintComponent(Graphics g) {
			PaintNode node = adjustSize(g);
			painter(g,node,new Point(hSpace,vSpace),null);
		}

		PaintNode adjustSize(Graphics g) {
			paintList.clear();
			PaintNode tree = calcSize(g,root);
			int ww = tree.w + hSpace*2;
			int hh = tree.h + vSpace*2;
			
			Dimension d = getSize();
			if (d.width == ww && d.height == hh) {
				return tree;
			}
			setPreferredSize(new Dimension(ww,hh));
			return tree;
		}

		PaintNode calcSize(Graphics g,TreeStructure node) {
			RichString name = new RichString(node.getTreeNodeExpression(),fontInfo);
			Dimension tdim = name.getSize(g);
	
			TreeStructure [] children = node.getTreeNodes();
			Dimension dim = new Dimension(0,0);
			PaintNode [] nodes = null;
			if (children != null && children.length > 0) {
				nodes = new PaintNode[children.length];
				for (int i=0;i<children.length;i++) {
					PaintNode r = calcSize(g,children[i]);
					nodes[i] = r;
					dim.width += r.w + hSpace;
					if (dim.height < r.h)
						dim.height = r.h;
				}
				dim.height += tdim.height + vSpace;
			} else {
				dim = new Dimension(tdim.width,tdim.height);
			}
			return new PaintNode(name,dim.width,dim.height,nodes);
		}

		/**
		   @param node current node
		   @param ps current tree left-upper position
		*/
		void painter(Graphics g, PaintNode node, Point ps, Point con) {
			Dimension tdim = node.name.getSize(g);
			//current node left-upper position
			int mx = ps.x + node.w/2 - tdim.width/2;
			int my = ps.y;
			if (node.children != null) {
				//modulation
				int avx = 0;
				int xx = 0;
				for (int i=0;i<node.children.length;i++) {
					avx += ps.x + xx + node.children[i].w/2;
					xx += node.children[i].w + hSpace;
				}
				mx = avx / node.children.length - tdim.width/2;
			} 
			if (con != null)
				g.drawLine(mx + tdim.width/2,my, con.x,con.y);
			node.name.drawContents(g,mx,my);
			Point myCon = new Point(mx+ tdim.width/2, my+tdim.height);
			if (node.children != null) {
				int xx = 0;
				for (int i=0;i<node.children.length;i++) {
					int cx = ps.x + xx;
					int cy = ps.y + tdim.height + vSpace;
					painter(g,node.children[i],new Point(cx,cy),myCon);
					xx += node.children[i].w + hSpace;
				}
			}
		}

		class PaintNode {
			PaintNode (RichString n,int w,int h,PaintNode [] ns) {
				this.name = n;
				this.w = w;this.h = h;
				this.children = ns;
			}
			RichString name;
			int x,y;
			int w,h;
			PaintNode [] children;
		}

	}

	public static void main(String[] args) {
		TreeStructure root = TreeStructureUtil.getTestTreeObject(30);
		TreeStructureViewer viewer = new TreeStructureViewer(root);
	}
}


