/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.core;

import jp.gr.java_conf.ccs2.util.StringUtil;



/**
 * This class manages application options.
 *
 * @author <a href="mailto:m.sakurai@dream.com">SAKURAI Masashi</a>
 * @version 1.0
 */
public abstract class AppConfig {

	private AppContext appContext;

	abstract protected Object getOptionObject(String key);

	abstract public String[] getOptionKeys();

	public abstract String [] getArgumentFiles();

	public void setAppContext(AppContext app) {
		this.appContext = app;
	}

	public AppContext getAppContext() {
		return appContext;
	}

	public String getDumpProperties() {
		StringBuffer sb = new StringBuffer();
		String[] keys = getOptionKeys();
		for (int i=0;i<keys.length;i++){
			String cont = getOptionObject(keys[i]).toString();
			sb.append(keys[i]+":"+cont+"\n");
		}
		String[] files = getArgumentFiles();
		for (int i=0;i<files.length;i++) {
			sb.append("FILE:"+files[i]+"\n");
		}
		return sb.toString();
	}

	/**
	   @param optionName option name that you want to get
	   @return option boolean. if wrong format or no option input, return false.
	 */
	public boolean getOption(String optionName) {
		return getOption(optionName,false);
	}

	/**
	   @param optionName option name that you want to get
	   @param def defualt value
	   @return option boolean. if wrong format or no option input, return defualt value.
	 */
	public boolean getOption(String optionName,boolean def) {
		Object obj = getOptionObject(optionName);
		if (obj == null || !(obj instanceof Boolean)) return def;
		return ((Boolean)obj)==Boolean.TRUE ? true:false;
	}
	
	/**
	   @param optionName option name that you want to get
	   @return option boolean. if wrong format or no option input, return null.
	 */
	public String getOptionString(String optionName) {
		return getOptionString(optionName,null);
	}

	/**
	   @param optionName option name that you want to get
	   @param def defualt value
	   @return option boolean. if wrong format or no option input, return defualt value.
	 */
	public String getOptionString(String optionName,String def) {
		Object obj = getOptionObject(optionName);
		if (obj == null) return def;
		if (!(obj instanceof String)) return obj.toString();
		return (String)obj;
	}

	/**
	   @param optionName option name that you want to get
	   @return option boolean. if wrong format or no option input, return -1.
	 */
	public int getOptionInteger(String optionName) {
		return getOptionInteger(optionName,-1);
	}

	/**
	   @param optionName option name that you want to get
	   @param def defualt value
	   @return option boolean. if wrong format or no option input, return defualt value.
	 */
	public int getOptionInteger(String optionName,int def) {
		String ret = getOptionString(optionName);
		if (StringUtil.isNull(ret)) return def;
		try {
			return Integer.parseInt(ret);
		} catch (NumberFormatException e) {
			if (appContext != null) {
				appContext.getMonitor().warning
					("property ["+optionName+"] is wrong number.");
			}
		}
		return def;
	}

	/**
	   @param optionName option name that you want to get
	   @return option boolean. if wrong format or no option input, return -1.
	 */
	public double getOptionDouble(String optionName) {
		return getOptionDouble(optionName,-1);
	}

	/**
	   @param optionName option name that you want to get
	   @param def defualt value
	   @return option boolean. if wrong format or no option input, return defualt value.
	 */
	public double getOptionDouble(String optionName,double def) {
		String ret = getOptionString(optionName);
		if (StringUtil.isNull(ret)) return def;
		try {
			return Double.parseDouble(ret);
		} catch (NumberFormatException e) {
			if (appContext != null)
				appContext.getMonitor().warning
					("property ["+optionName+"] is wrong number.");
		}
		return def;
	}

	public String getArgumentFile() {
		if (getArgumentFiles().length == 0) return null;
		return getArgumentFiles()[0];
	}

}
