/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.core;
import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;

import jp.gr.java_conf.ccs2.util.FileRotator;
import jp.gr.java_conf.ccs2.util.StringUtil;


/** Record and show a message. */
public class DefaultMonitor implements MessageMonitor {

	private static String separator = System.getProperty("line.separator");

	private String logFileName;
	private PrintStream out;
	private String encoding;

	private FileRotator rotator;

	private int debugFlushLevel = ERROR;
	private int debugRecordNumber = 0;
	private LinkedList debugList = new LinkedList();

	private int logLevel = NORMAL;
	private String format = FORMAT_DATE+"|"+FORMAT_LEVEL+"|"+FORMAT_MESSAGE;
	private String [] filters ;

	private DateFormat dateFormatter = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	/** construct with given log level and logfile 
		@param filename log filename. if null, no log-file.
		@param logLevel log level.
		@param out console object, such as System.out or System.err.
	*/
	public DefaultMonitor(String filename,int logLevel,PrintStream out) {
		setLogFile(filename);
		if (logLevel < DEBUG || logLevel > ERROR) {
			logLevel = NORMAL;
		}
		this.logLevel = logLevel;
		this.out = out;
	}
	/** construct with NORMAL log level,
		no logfile and STDOUT output. */
	public DefaultMonitor() {
		this(null,NORMAL,System.out);
	}
	/** construct with given log level, no logfile
		and STDOUT output.
	*/
	public DefaultMonitor(int logLevel) {
		this(null,logLevel,System.out);
	}
	/** construct with NORMAL log level,
		given logfile and no output.
	*/
	public DefaultMonitor(String filename) {
		this(filename,NORMAL,null);
	}
	/** construct with AppConfig.
	 */
	public DefaultMonitor(AppConfig config) {
		setLogFile(config.getOptionString("logfile"));
		if (logFileName == null) this.out = System.out;
		setLogLevel( config.getOptionString("logLevel") );
		setFilters( config.getOptionString("logFilters") );
		setEncoding( config.getOptionString("logEncoding"));
	}

	public void setDebugFlushLevel(int a) {
		if (a > DEBUG && a <= ERROR) {
			debugFlushLevel = a;
		}
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public void setDebugRecordNumber(int i) {
		if (i >= 0) {
			debugRecordNumber = i;
		}
	}

	public void setRotator(FileRotator rotator) {
		this.rotator = rotator;
		rotator.setFilename(logFileName);
	}

	public void setLogFile(String filename) {
		if (!StringUtil.isNull(filename)) {
			logFileName = filename;
			if (rotator != null) {
				rotator.setFilename(filename);
			}
		}
	}

	public void setDateFormatter(DateFormat df) {
		this.dateFormatter = df;
	}

	public void setFilters(String args) {
		if (StringUtil.isNull(args)) return;
		setFilters(StringUtil.split(args));
	}
	public void setFilters(String [] args) {
		filters = args;
	}
	public int getLogLevel() {return logLevel;}
	public void setLogLevel(int a) {
		if (a >= DEBUG && a <= ERROR) {
			logLevel = a;
		}
	}
	public void setLogLevel(String in) {
		for(int i=0;i<LEVELS.length;i++) {
			if (LEVELS[i].equalsIgnoreCase(in)) {
				setLogLevel(i);
				return;
			}
		}
		setLogLevel(NORMAL);
	}
	public PrintStream getOutput() {return out;}
	public void setOutput(PrintStream p) {out = p;}
	public void setFormat(String f) {format = f;}
	public String toString() {
		String ret = "CLASS:"+getClass().getName()+"\n";
		ret += "OUTPUT:"+ ((getOutput()!=null) 
						   ? getOutput().getClass().getName():null)+"\n";
		ret +="LOGFILE:"+logFileName+"\n";
		ret +="STATE:"+LEVELS[getLogLevel()];
		return ret;
	}

	public void debug(String message) {
		message(message,DEBUG);
	}
	public void verbose(String message) {
		message(message,VERBOSE);
	}
	public void normal(String message) {
		message(message,NORMAL);
	}
	public void warning(String message) {
		message(message,WARNING);
	}
	public void error(String message) {
		message(message,ERROR);
	}

	public void debugStackTrace(String message) {
		if (blockRecording(message)) return;
		Throwable t = new Throwable("DEBUG trace");
		StringWriter sw = new StringWriter();
		t.printStackTrace(new PrintWriter(sw));
		sw.flush();
		String trace = sw.toString()+"\n";
		message(message+trace,DEBUG);
	}

	public void recordStackTrace(Throwable t) {
		String trace = "";
		StringWriter sw = new StringWriter();
		t.printStackTrace(new PrintWriter(sw));
		sw.flush();
		trace = sw.toString();
		message(trace,ERROR);
	}

	public void message(int level,String m) {
		message(m,level);
	}

	protected boolean blockRecording(String message) {
		if (filters != null && filters.length!=0) {
			for (int i=0;i<filters.length;i++) {
				if (message.indexOf(filters[i])!=-1) {
					return false;
				}
			}
			return true;
		}
		return false;
	}

	protected synchronized void message(String message,int level) {
		if (logLevel > level) {
			if (debugRecordNumber > 0) {
				debugList.add(formatMessage(message,level));
				while (debugList.size() > debugRecordNumber) {
					debugList.removeFirst();
				}
			}
			return;
		}
		if (blockRecording(message)) return;
		String er = formatMessage(message,level);
		if (out != null) {
			out.println(er);
		}
		if (logFileName != null) {
			try {
				Writer logFile = null;
				if (StringUtil.isNull(encoding)) {
					logFile = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(logFileName,true)));
				} else {
					logFile = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(logFileName,true),encoding));
				}
				logFile.write(er + separator);
				if ( (level >= debugFlushLevel) &&
					 (debugRecordNumber > 0) && (debugList.size() > 0) ) {
					logFile.write("===(Debug record begin)==="+separator);
					while(debugList.size() > 0) {
						logFile.write(debugList.removeFirst() + separator);
					}
					logFile.write("===(Debug record end)====="+separator);
				}
				logFile.flush();
				logFile.close();
				if (rotator != null) {
					rotator.rotate();
				}
			} catch(IOException e) {
				System.err.println("fatal error : IOException at DefualtMonitor.");
				System.err.println(er);
			}
		}
	}

	private String formatMessage(String message,int level) {
		String ret = StringUtil.replace(format,FORMAT_LEVEL,LEVELS[level]);
		ret = StringUtil.replace(ret,FORMAT_DATE,dateFormatter.format(new Date()));
		return StringUtil.replace(ret,FORMAT_MESSAGE,message);
	}
	
	//=====================================

	public static final String defaultName = "ccslog.txt";
	protected static MessageMonitor defaultMonitor;

	protected static MessageMonitor getDefaultMonitor() {
		if (defaultMonitor == null) {
			defaultMonitor = new DefaultMonitor(defaultName,NORMAL,System.out);
		}
		return defaultMonitor;
	}

	public static void errorLog(String message) {
		getDefaultMonitor().error(message);
	}
	public static void warningLog(String message) {
		getDefaultMonitor().warning(message);
	}
	public static void messageLog(String message) {
		getDefaultMonitor().normal(message);
	}

}
