/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.sql;

import java.sql.*;

public class DBAccessContext {

	private Connection connection;
	private String preparedSql = null;

	/**
	   you should initialize the connection employing "initConnection".
	 */
	public DBAccessContext() {
	}

	public DBAccessContext(Connection connection) {
		initConnection(connection);
	}

	public void initConnection(Connection con) {
		this.connection = con;
	}

	/**
	   @param sql if null, this context make a normal statement object.
	              if not null, this context give a prepared statement object
				  to client.
	 */
	public void setPreparedSql(String sql) {
		preparedSql = sql;
	}

	private Statement getStatement() throws SQLException {
		if (preparedSql == null) {
			return connection.createStatement();
		} else {
			return connection.prepareStatement(preparedSql);
		}
	}

	/**
	   Access DB and close the Statement.
	   This method does not close the connection, so you
	   can re-use this context object.At last, 
	   you should call closeConnection method.
	   @param client DBAccess object
	*/
	public void doAccess(DBAccess client) {
		Statement stmt = null;
		try {
			stmt = getStatement();
			client.access(stmt);
		} catch (SQLException e) {
			client.exceptionHandler(e);
		} finally {
			try {
				if (stmt != null)
					stmt.close();
			} catch (SQLException e) {
				client.exceptionHandler(e);
			}
		}
	}

	/**
	   Access DB and close the Statement and Connection object.
	   @param client DBAccess object
	*/
	public void doAccessAtOnce(DBAccess client) {
		doAccessAtOnce(new DBAccess [] {client});
	}

	/**
	   Access DB and close the Statement and Connection object.
	   After calling this method, you can not use this DBAccessContext
	   object without calling "initConnection", 
	   because the connection is closed.

	   @param clients DBAccess object
	*/
	public void doAccessAtOnce(DBAccess [] clients) {
		if (clients == null || clients.length == 0) return;
		try {
			for (int i=0;i<clients.length;i++) {
				doAccess(clients[i]);
			}
		} finally {
			try {
				if (connection != null && 
					(!connection.isClosed())) {
					connection.close();
				}
			} catch (SQLException e) {
				clients[0].exceptionHandler(e);
			}
		}
	}

	/**
	   this method provides exclusive transaction task, 
	   such as multi-table INSERT, UPDATE command and complecated task.
	   first, turns off auto commit mode, locks tables and begins transaction.
	   after your code, send commit signal and close all connection.
	   if some exception is occured during your code, the task
	   is rolled back and exceptionHandler is called.

	   After calling this method, you can not use this DBAccessContext
	   object without calling "initConnection", 
	   because the connection is closed.

	   @param client DB access object
	   @param tables tables to lock
	 */
	public void doTransactionAccessAtOnce(DBAccess client,String [] tables) {
		doTransactionAccessAtOnce(new DBAccess[] {client}, tables);
	}

	/**
	   this method provides exclusive transaction task, 
	   such as multi-table INSERT, UPDATE command and complecated task.
	   first, turns off auto commit mode, locks tables and begins transaction.
	   after your code, send commit signal and close all connection.
	   if some exception is occured during your code, the task
	   is rolled back and exceptionHandler is called.

	   In case of exception, the exception handler of the first client object
	   is called.

	   @param clients DB access object
	   @param tables tables to lock
	 */
	public void doTransactionAccess(DBAccess [] clients,String [] tables) {
		if (clients == null || clients.length == 0) return;
		Statement stmt = null;
		NativeDBCommand command = null;
		try {
			connection.setAutoCommit(false);
			command = JDBCUtil.getNativeDBCommand(connection);
			stmt = getStatement();
			command.lockTables(stmt,tables,true);
			for (int i=0;i<clients.length;i++) {
				clients[i].access(stmt);
			}
			command.unlockTables(stmt,tables);
			connection.commit();
		} catch (SQLException e) {
			if (command != null && stmt != null) {
				try {
					connection.rollback();
				} catch (SQLException ex) {
					clients[0].exceptionHandler(ex);
				}
			}
			clients[0].exceptionHandler(e);
		} finally {
			try {
				if (stmt != null ) {
					stmt.close();
				}
			} catch (SQLException e) {
				clients[0].exceptionHandler(e);
			}
		}
	}

	/**
	   this method provides exclusive transaction task, 
	   such as multi-table INSERT, UPDATE command and complecated task.
	   first, turns off auto commit mode, locks tables and begins transaction.
	   after your code, send commit signal and close all connection.
	   if some exception is occured during your code, the task
	   is rolled back and exceptionHandler is called.

	   After calling this method, you can not use this DBAccessContext
	   object without calling "initConnection", 
	   because the connection is closed.

	   In case of exception, the exception handler of the first client object
	   is called.

	   @param clients DB access object
	   @param tables tables to lock
	 */
	public void doTransactionAccessAtOnce(DBAccess [] clients,String [] tables) {
		try {
			doTransactionAccess(clients,tables);
		} finally {
			try {
				if (connection != null && (!connection.isClosed())) {
					connection.close();
				}
			} catch (SQLException e) {
				clients[0].exceptionHandler(e);
			}
		}
	}

	/**
	   this method provides share locking task, 
	   such as multi-table and complecated query task.
	   locks tables before your code.
	   after your task, unlock tables and close all connection.
	   if some exception is occured during your code,
	   exceptionHandler is called.

	   After calling this method, you can not use this DBAccessContext
	   object without calling "initConnection", 
	   because the connection is closed.

	   In case of exception, the exception handler of the first client object
	   is called.

	   @param client DB access object
	   @param tables tables to lock
	 */
	public void doLockedAccessAtOnce(DBAccess client,String [] tables) {
		Statement stmt = null;
		NativeDBCommand command = null;
		try {
			command = JDBCUtil.getNativeDBCommand(connection);
			stmt = getStatement();
			connection.setAutoCommit(false);
			command.lockTables(stmt,tables,false);
			client.access(stmt);
			command.unlockTables(stmt,tables);
			connection.commit();
		} catch (SQLException e) {
			if (command != null && stmt != null) {
				try {
					connection.rollback();
				} catch (SQLException ex) {
					client.exceptionHandler(ex);
				}
			}
			client.exceptionHandler(e);
		} finally {
			try {
				if (stmt != null ) {
					stmt.close();
				}
				if (connection != null && (!connection.isClosed()) ) {
					connection.close();
				}
			} catch (SQLException e) {
				client.exceptionHandler(e);
			}
		}
	}

	/**
	   close connection.
	   Then, if SQLException occured, client.exceptionHandler() is called.

	 */
	public void closeConnection(DBAccess client) {
		try {
			connection.close();
		} catch (SQLException e) {
			client.exceptionHandler(e);
		}
	}

	/**
	   close connection.
	   Then, if SQLException occured, track trace is dumped.
	 */
	public void closeConnection() {
		try {
			connection.close();
		} catch (SQLException e) {
			e.printStackTrace();
		}
	}
	
}


