/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.sql;
import java.sql.*;
import java.util.Map;
import jp.gr.java_conf.ccs2.util.StringUtil;
import java.util.ArrayList;
import jp.gr.java_conf.ccs2.util.StringFilter;
import java.util.HashMap;
import java.util.Iterator;

/**
   This utility class provides easy access with Connection and SQL, and
   easy format for string results.
*/
public class SQL2String {

	/**
	   Utility for one result query, such as "SELECT max(id) FROM atable".
	   This method close the given connection after DB access.
	 */
	public static String getResult(Connection con,String sql) throws SQLException {
		final String fsql = sql;
		final String [] ret = new String[1];
		final DBAccess access = new DBAccess() {
				public void access(Statement stat) throws SQLException {
					ResultSet set = stat.executeQuery(fsql);
					if (set != null && set.next()) {
						ret[0] = set.getString(1);
					}
				}
				public void exceptionHandler(SQLException e) {
					e.printStackTrace();
				}
			};
		DBAccessContext ct = new DBAccessContext(con);
		ct.doAccessAtOnce(access);
		return ret[0];
	}

	/**
	   Utility for SQL access and formatting a result.
	   This method close the given connection after DB access.
	 */
	public static String [] getLayoutedResults(Connection con,String sql,String layout) throws SQLException {
		return getLayoutedResults(con,sql,layout,null);
	}

	/**
	   Utility for SQL access and formatting a result.
	   This method close the given connection after DB access.
	 */
	public static String [] getLayoutedResultsCont(Connection con,String sql,String layout) throws SQLException {
		return getLayoutedResultsCont(con,sql,layout,null);
	}

	/**
	   Utility for SQL access and formatting a result.
	   This method closes the given connection after DB access.
	 */
	public static String [] getLayoutedResults(Connection con,String sql,String layout,Map transformers) throws SQLException {
		LocalAccess la = new LocalAccess(transformers,sql,layout);
		DBAccessContext ct = new DBAccessContext(con);
		ct.doAccessAtOnce(la);
		return la.getResultString();
	}

	/**
	   Utility for SQL access and formatting a result.
	   This method doesn't close the given connection after DB access.
	 */
	public static String [] getLayoutedResultsCont(Connection con,String sql,String layout,Map transformers) throws SQLException {
		LocalAccess la = new LocalAccess(transformers,sql,layout);
		DBAccessContext ct = new DBAccessContext(con);
		ct.doAccess(la);
		return la.getResultString();
	}

	//===================================
	// Layout Data
	//===================================
	
	public static String CTRL = "%";

	public static String [] layoutData(String layout,ResultSet rs) throws SQLException {
		return layoutData(layout,rs,null);
	}

	public static String [] layoutData(String layout,ResultSet rs,Map map) throws SQLException {
		if (StringUtil.isNull(layout)) return null;
		if (rs == null) {
			return clearLayoutDatas(layout);
		}
		String [] columnNames = makeColumnArray(rs);
		ArrayList rows = new ArrayList();
		for (int row=0;rs.next();row++) {
			String line = layout;
			for (int i=0;i<columnNames.length;i++) {
				line = transformWithMap(line,rs.getString(i+1),
										columnNames[i],map);
			}
			rows.add(line);
		}
		String [] rets = new String[rows.size()];
		return (String[])rows.toArray(rets);
	}

	/**
	   Layout one line from HashMap object.
	 */
	public static String layoutData(String layout,HashMap rs,Map map)  {
		if (StringUtil.isNull(layout)) return null;
		if (rs == null || rs.size() == 0) {
			String [] ret = clearLayoutDatas(layout);
			return ret[0];
		}
		Iterator it = rs.keySet().iterator();
		String line = layout;
		while(it.hasNext()) {
			String colName = (String)it.next();
			String val = null;
			Object obj = rs.get(colName);
			if (obj instanceof String) {
				val = (String)obj;
			} else {
				if (obj == null) val = "";
				else val = obj.toString();
			}
			line = transformWithMap(line,val,colName,map);
		}
		return line;
	}

	private static String transformWithMap(String line,
										   String data,
										   String columnName,
										   Map map) {
		if (map != null) {
			StringFilter all =
				(StringFilter)map.get("*");
			if (all != null) {
				data = all.applies(data);
			}
			StringFilter tr = 
				(StringFilter)map.get(columnName);
			if (tr != null) {
				data = tr.applies(data);
			}
		}
		return StringUtil.replace(line,columnCtrlCode(columnName),data);
	}

	private static String columnCtrlCode(String clmName) {
		return CTRL+clmName+CTRL;
	}

	private static String [] makeColumnArray(ResultSet rs) throws SQLException {
		ResultSetMetaData rsmd = rs.getMetaData();
		int cols = rsmd.getColumnCount();
		String [] columnNames = new String[cols];
		for (int i=0;i<cols;i++) {
			columnNames[i] = rsmd.getColumnName(i+1).toLowerCase();
		}
		return columnNames;
	}

	private static String [] clearLayoutDatas(String in) {
		String [] ret = {clearLayoutData(in)};
		return ret;
	}

	private static char [] NONCTRL = {' ','\n','\r','"','<','>','\'','=','(',')','-','$','&','+','*','[',']','{','}',',','/',';',':','@','\t','!',};

	private static String clearLayoutData(String in) {
		StringBuffer ret = new StringBuffer();
	outer:
		while (true) {
			int ps = in.indexOf(CTRL);
			if (ps == -1) break;
			int pe = in.indexOf(CTRL,ps+1);
			if (pe == -1) break;
			String ctrlStr = in.substring(ps,pe+1);
			for (int i=0;i<NONCTRL.length;i++) {
				if (ctrlStr.indexOf(NONCTRL[i]) != -1) {
					ret.append(in.substring(0,pe));
					in = in.substring(pe);
					continue outer;
				}
			}
			ret.append(in.substring(0,ps));
			in = in.substring(pe+1);
		}
		ret.append(in);
		return ret.toString();
	}
	
	
}

class LocalAccess implements DBAccess {
	
	String [] results;
	Map ltransformers;
	String sql,layout;
	
	LocalAccess(Map transformers,String asql,String alayout) {
		ltransformers = transformers;
		sql = asql;
		layout = alayout;
	}

	String[] getResultString() {
		return results;
	}
	
	public void access(Statement stat) throws SQLException {
		ResultSet set = stat.executeQuery(sql);
		results = SQL2String.layoutData(layout,set,ltransformers);
	}
	
	public void exceptionHandler(SQLException e) {
		e.printStackTrace();
	}
}

