/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.util;
import java.util.*;
import java.lang.ref.*;

/** 
	This class holds objects as like HashMap.
	The value objects will be collected by GC.
	If the number of objects in this collection reaches the maxSize, 
	the objects that are not used much frequently are cleared.
	This class is not thread-safe.
 */

public class CacheHashMap implements Map {

	private HashMap referenceCache = new HashMap();
	private LinkedList keyList = new LinkedList();
	private int maxSize = 0;

	public CacheHashMap(int maxSize) {
		setMaxSize(maxSize);
	}

	public void setMaxSize(int sz) {
		this.maxSize = sz;
		clear();
	}

	public void clear() {
		keyList.clear();
		referenceCache.clear();
	}

	private void refresh() {
		Iterator it = keyList.iterator();
		while(it.hasNext()) {
			Object key = it.next();
			SoftReference wr = (SoftReference)referenceCache.get(key);
			Object obj = wr.get();
			if (obj == null) {
				it.remove();
				referenceCache.remove(key);
			}
		}
	}

	public boolean containsKey(Object key) {
		refresh();
		return referenceCache.containsKey(key);
	}

	public boolean containsValue(Object value) {
		refresh();
		Iterator it = keyList.iterator();
		while(it.hasNext()) {
			Object key = it.next();
			SoftReference wr = (SoftReference)referenceCache.get(key);
			Object obj = wr.get();
			if (obj != null && value.equals(obj)) {
				return true;
			}
		}
		return false;
	}

	class SoftEntry implements Map.Entry {
		private SoftReference ref;
		private Object key;
		SoftEntry(Object tkey,SoftReference wr) {
			key = tkey;
			ref = wr;
		}
		public Object getKey() {
			return key;
		}
		public Object getValue() {
			return ref.get();
		}
		public Object setValue(Object value) {
			throw new InternalError("Not supported operation.");
		}
	}

	public Set entrySet() {
		Iterator it = keyList.iterator();
		TreeSet set = new TreeSet();
		while(it.hasNext()) {
			Object key = it.next();
			SoftReference wr = (SoftReference)referenceCache.get(key);
			set.add(new SoftEntry(key,wr));
		}
		return set;
	}

	public boolean equals(Object o) {
		return this == o;
	}

	public Object get(Object key) {
		SoftReference wr = (SoftReference)referenceCache.get(key);
		if (wr == null) {
			return null;
		}
		Object obj = wr.get();
		if (obj == null) {
			remove(key);
			return null;
		}
		keyList.remove(key);
		keyList.addFirst(key);
		return obj;
	}

	public boolean isEmpty() {
		refresh();
		return keyList.isEmpty();
	}

	public Set keySet() {
		refresh();
		return referenceCache.keySet();
	}

	public Object put(Object key, Object value) {
		while(keyList.size() >= maxSize) {
			Object lk = keyList.removeLast();
			referenceCache.remove(lk);
		}
		keyList.addFirst(key);
		SoftReference wr = new SoftReference(value);
		SoftReference lwr = (SoftReference)referenceCache.put(key,wr);
		if (lwr != null) {
			return lwr.get();
		}
		return null;
	}

	public void putAll(Map t) {
		Iterator it = t.keySet().iterator();
		while(it.hasNext()) {
			Object key = it.next();
			put(key,t.get(key));
		}
	}

	public Object remove(Object key) {
		keyList.remove(key);
		SoftReference wr = (SoftReference)referenceCache.remove(key);
		return wr.get();
	}

	public int size() {
		refresh();
		return keyList.size();
	}

	public Collection values() {
		Iterator it = referenceCache.entrySet().iterator();
		Collection set = new LinkedList();
		while(it.hasNext()) {
			SoftReference wr = (SoftReference)it.next();
			Object obj = wr.get();
			if (obj != null) {
				set.add(obj);
			}
		}
		return set;
	}

	public static void main(String [] args) {
		CacheHashMap map = new CacheHashMap(4);
		Integer i0 = new Integer(0);
		map.put(i0,new Integer(100));
		Integer i1 = new Integer(1);
		map.put(i1,new Integer(200));
		map.put(new Integer(2),new Integer(300));
		map.put(new Integer(3),new Integer(400));
		System.out.println("size:"+map.size());
		System.out.println("0:"+map.get(new Integer(0)));
		map.put(new Integer(4),new Integer(500));
		System.out.println("1:"+map.get(i1));
		System.gc();
		System.out.println("size:"+map.size());
	}
	
}

