/* CCS, Class Collection by Sakurai
 *	Copyright (C) 2000-2001 SAKURAI, Masashi (m.sakurai@cmt.phys.kyushu-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.ccs2.util;
import java.io.*;
import java.util.*;


/** Pick up all files from sub-directories */
public class LSRFiles {

	private String fs = System.getProperty("file.separator");
	private boolean subdir = true;
	private Vector files = new Vector();

	/** build the list of all files below the given path.
	 * @param path direcotry. if null, current directory will be set.
	 */
	public LSRFiles(String path) throws IOException {
		this(path,true);
	}

	/** build the list of all files from the given path.
	 * @param path direcotry. if null, current directory will be set.
	 * @param subdir if true, pick up from whole sub-directories.
	 */
	public LSRFiles(String path,boolean subdir) throws IOException {
		if (path == null || path.equals("")) {
			path = System.getProperty("user.dir");
		}
		this.subdir = subdir;
		initFiles(path);
	}

	/** build the list of all files below the given path. 
	 * (Non-exception version)
	 * @param path direcotry.
	 * if null, current directory will be set.
	 */
	public static LSRFiles getLSR(String path) {
		LSRFiles lsr = null;
		try {
			lsr = new LSRFiles(path);
		} catch(IOException ex) {
			ex.printStackTrace();
		}
		return lsr;
	}

	/** build a list of all files below the given path
	 * @param path direcotry or file path 
	 */
	private void initFiles(String path) throws IOException {
		File tfile = new File(path);
		if (tfile.isFile()) {
			//file
			files.addElement(path);
		} else {
			//directory
			Enumeration entries = getFilesByDir(path);
			if (entries != null) {
				while (entries.hasMoreElements()) {
					files.addElement((String)entries.nextElement());
				}
			}
		}
		if (files.size() == 0) {
			System.err.println("file or directory not found.");
		}
	}

	private Enumeration getFilesByDir(String absolutePath) throws IOException {
		//System.out.println("--> "+absolutePath);
		Vector tempList = new Vector();
		File dir = new File(absolutePath);
		if (!dir.isDirectory()) {
			return null;
		}
		String [] entries = dir.list();
		//check the sub-directory
		for (int i=0;i<entries.length;i++) {
			String name = absolutePath + fs + entries[i];
			File tfile = new File( name );
			if ( tfile.isFile() )
				tempList.addElement( tfile.getAbsolutePath() );
			else {
				if (!subdir) continue;
				Enumeration e = getFilesByDir( name );
				if (e != null) {
					while (e.hasMoreElements()) {
						tempList.addElement( (String)e.nextElement() );
					}
				} 
			}
		}
		return tempList.elements();
	}

	public void setFilenameFilter(FilenameFilter ff) {
		File [] tfiles = getFiles();
		if (tfiles == null) return;
		Vector newFiles = new Vector();
		for (int i=0;i<tfiles.length;i++) {
			File curFile = tfiles[i];
			String sdir = curFile.getParent();
			if (sdir == null) {
				curFile = new File( curFile.getAbsolutePath() );
				sdir = curFile.getParent();
			}
			if (ff.accept(new File(sdir),curFile.getName()))
				newFiles.addElement(curFile.getPath());
		}
		files = newFiles;
	}

	public String [] getFilenames() {
		if (files.isEmpty()) return null;
		Enumeration e = files.elements();
		String [] filenames = new String [files.size()];
		for (int i=0; e.hasMoreElements() ;i++) {
			filenames[i] = (String)e.nextElement();
		}
		return filenames;
	}

	public File [] getFiles() {
		String [] filenames = getFilenames();
		if (filenames == null) return null;
		File [] tfiles = new File[filenames.length];
		for (int i=0;i<filenames.length;i++)
			tfiles[i] = new File(filenames[i]);
		return tfiles;
	}

	// test method
	public static void main(String [] args) {
		String dir = null;
		if (args.length != 0)
			dir = args[0];
		System.out.println("getting from ["+dir+"]");
		LSRFiles lsr = LSRFiles.getLSR( dir );
		String [] list = lsr.getFilenames();
		for (int i=0;i<list.length;i++)
			System.out.println(list[i]);
	}
}

